package tdapi

import (
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"path"
	"time"

	"github.com/opentracing/opentracing-go"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/logging/yt/service"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/metrics"
)

type Client interface {
	InitSearch(initSearchRequest *InitSearchRequest)
}

type TicketDaemonAPIClient struct {
	httpClient        *http.Client
	baseURL           string
	appLogger         log.Logger
	serviceCallLogger *service.Logger
}

func NewTicketDaemonAPIClient(
	httpClient *http.Client,
	baseURL string,
	appLogger log.Logger,
	serviceCallLogger *service.Logger,
) *TicketDaemonAPIClient {
	return &TicketDaemonAPIClient{
		httpClient:        httpClient,
		baseURL:           baseURL,
		appLogger:         appLogger,
		serviceCallLogger: serviceCallLogger,
	}
}

const (
	initSearchPath   = "jsendapi/front/init_search_by_query"
	serviceName      = "ticket_daemon"
	initSearchMethod = "init_search"
)

func (ticketDaemonAPIClient *TicketDaemonAPIClient) InitSearch(initSearchRequest *InitSearchRequest) {
	initSearchURL, _ := url.Parse(ticketDaemonAPIClient.baseURL)
	initSearchURL.Path = path.Join(initSearchURL.Path, initSearchPath)
	initSearchURL.RawQuery = ticketDaemonAPIClient.setService(initSearchRequest.ToURLValues()).Encode()

	request, _ := http.NewRequest(http.MethodGet, initSearchURL.String(), nil)
	if span := opentracing.SpanFromContext(initSearchRequest.Context); span != nil {
		_ = opentracing.GlobalTracer().Inject(
			span.Context(),
			opentracing.HTTPHeaders,
			opentracing.HTTPHeadersCarrier(request.Header),
		)
	}
	start := time.Now()
	response, err := ticketDaemonAPIClient.httpClient.Do(request)
	requestDuration := time.Since(start)
	defer func() {
		ticketDaemonAPIClient.serviceCallLogger.Log(
			service.NewRecord(
				initSearchRequest.QueryParameters.JobID,
				serviceName,
				initSearchMethod,
				requestDuration,
				err != nil || response.StatusCode != http.StatusOK,
			),
		)
	}()
	metrics.GlobalWizardMetrics().TicketDaemonAPIInitSearchTimer.RecordDuration(requestDuration)

	if err != nil {
		ticketDaemonAPIClient.appLogger.Error(
			fmt.Sprintf("couldn't send init_search request to ticked-daemon-api: %v", err),
			log.String("job_id", initSearchRequest.QueryParameters.JobID),
		)
		return
	}
	defer response.Body.Close()

	if response.StatusCode != http.StatusOK {
		responseBody, _ := ioutil.ReadAll(response.Body)
		ticketDaemonAPIClient.appLogger.Error(
			fmt.Sprintf(
				"request to ticked-daemon-api has finished with status code %v. response: %v",
				response.Status,
				string(responseBody),
			),
			log.String("job_id", initSearchRequest.QueryParameters.JobID),
		)
	}
}

func (ticketDaemonAPIClient *TicketDaemonAPIClient) setService(values url.Values) url.Values {
	values.Set("service", "unisearch")
	return values
}
