package settings

import (
	"context"
	"time"

	"a.yandex-team.ru/travel/avia/library/go/probes"
	wizardgeobase "a.yandex-team.ru/travel/avia/wizard/pkg/geobase"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/dynamicresources"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/services/featureflag"
	"a.yandex-team.ru/travel/library/go/configuration"
)

type Settings struct {
	Logging                         *LoggingSettings
	Metrics                         *MetricsSettings
	YdbSettings                     *YdbSettings `yaml:"ydb"`
	Conversion                      *ConversionSettings
	References                      *ReferencesSettings
	Personalization                 *PersonalizationSettings
	DynamicResources                dynamicresources.Config `yaml:"dynamic_resources"`
	Search                          *SearchSettings
	GOGC                            int           `config:"AVIA_WIZARD_GOGC" yaml:"gogc"`
	Port                            uint16        `config:"AVIA_WIZARD_PORT"`
	HTTPClientTimeout               time.Duration `yaml:"http_client_timeout"`
	Languages                       []string
	TankerKeysetPath                string                                              `config:"AVIA_WIZARD_TANKER_KEYSET_PATH" yaml:"tanker_keyset_path"`
	MaxStaticRequestDuration        time.Duration                                       `config:"AVIA_WIZARD_MAX_STATIC_REQUEST_DURATION" yaml:"max_static_request_duration"`
	MaxDynamicRequestDuration       time.Duration                                       `config:"AVIA_WIZARD_MAX_DYNAMIC_REQUEST_DURATION" yaml:"max_dynamic_request_duration"`
	MaxInflightRequests             uint32                                              `config:"AVIA_WIZARD_MAX_INFLIGHT_REQUESTS" yaml:"max_inflight_requests"`
	MaxVariantsInfoRequestDuration  time.Duration                                       `config:"AVIA_WIZARD_MAX_VARIANTS_INFO_REQUEST_DURATION" yaml:"max_variants_info_request_duration"`
	MaxVariantsInfoInflightRequests uint32                                              `config:"AVIA_WIZARD_MAX_VARIANTS_INFO_INFLIGHT_REQUESTS" yaml:"max_variants_info_inflight_requests"`
	SharedFlightsBaseURL            string                                              `config:"AVIA_WIZARD_SHARED_FLIGHTS_BASE_URL" yaml:"shared_flights_base_url"`
	BannedFlights                   string                                              `config:"AVIA_WIZARD_BANNED_FLIGHTS" yaml:"banned_flights"`
	FrontendURLs                    FrontendURLs                                        `yaml:"frontend_urls"`
	TravelPortalURLs                FrontendURLs                                        `yaml:"travel_portal_urls"`
	XRedirectURLs                   FrontendURLs                                        `yaml:"xredirect_urls"`
	YandexEnvironmentType           string                                              `config:"YANDEX_ENVIRONMENT_TYPE" yaml:"yandex_environment_type"`
	NearestSettlementMaxRadiusKM    float32                                             `yaml:"nearest_settlement_max_radius_km"`
	LanguageCaseFallbacks           map[models.LanguageCasePair]models.LanguageCasePair `yaml:"language_case_fallbacks"`
	LanguageFallbacks               map[models.Lang]models.Lang                         `yaml:"language_fallbacks"`
	RedirectSecretKey               string                                              `config:"AVIA_WIZARD_REDIRECT_SECRET_KEY" yaml:"redirect_secret_key"`
	TDApiRedirectSecretKey          string                                              `config:"AVIA_WIZARD_TD_API_REDIRECT_SECRET_KEY" yaml:"td_api_redirect_secret_key"`
	TicketDaemonAPIBaseURL          string                                              `config:"AVIA_WIZARD_TICKET_DAEMON_API_BASE_URL" yaml:"ticket_daemon_api_base_url"`
	DebugLogResponseJSON            bool                                                `config:"AVIA_WIZARD_DEBUG_LOG_RESPONSE_JSON" yaml:"debug_log_response_json"`
	GeobaseConfig                   *wizardgeobase.Config                               `yaml:"geobase"`
	Probes                          *probes.HTTPConfig
	FeatureFlag                     *featureflag.Config `yaml:"feature_flag"`
}

type PersonalizationSettings struct {
	Enabled            bool
	AppTimeout         time.Duration `config:"AVIA_WIZARD_PERSONALIZATION_TIMEOUT" yaml:"app_timeout"`
	TvmID              uint32        `config:"AVIA_WIZARD_PERSONALIZATION_TVM_ID" yaml:"tvm_id"`
	YPlannerID         string        `config:"AVIA_WIZARD_PERSONALIZATION_Y_PLANNER_ID" yaml:"y_planner_id"`
	ConnectionsPerHost uint          `config:"AVIA_WIZARD_PERSONALIZATION_CONNECTIONS_PER_HOST" yaml:"connections_per_host"`
}

var DefaultPersonalizationSettings = PersonalizationSettings{
	YPlannerID:         "avia-personalization-testing.api.grpc",
	AppTimeout:         75 * time.Millisecond,
	ConnectionsPerHost: 5,
}

func NewSettings() *Settings {
	loader := configuration.NewDefaultConfitaLoader()
	settings := Settings{
		GOGC:                            100,
		Logging:                         defaultLoggingSettings(),
		Metrics:                         defaultMetricsSettings(),
		YdbSettings:                     defaultYdbSettings(),
		References:                      defaultReferencesSettings(),
		Personalization:                 &DefaultPersonalizationSettings,
		DynamicResources:                dynamicresources.DefaultConfig,
		Search:                          defaultSearchSettings(),
		Port:                            8080,
		TicketDaemonAPIBaseURL:          "http://ticket-daemon-api.testing.avia.yandex.net",
		HTTPClientTimeout:               30 * time.Second,
		Languages:                       []string{"ru", "tr", "uk", "kk", "be"},
		TankerKeysetPath:                "./data/wizard_default_keyset.json",
		Conversion:                      defaultConversionSettings(),
		MaxStaticRequestDuration:        140 * time.Millisecond,
		MaxDynamicRequestDuration:       1 * time.Second,
		MaxInflightRequests:             50,
		MaxVariantsInfoInflightRequests: 50,
		SharedFlightsBaseURL:            "http://shared-flights.testing.avia.yandex.net",
		YandexEnvironmentType:           "development",
		FrontendURLs:                    getTestingFrontendURLs(),
		XRedirectURLs:                   getTestingXRedirectURLs(),
		TravelPortalURLs:                getTestingTravelPortalURLs(),
		NearestSettlementMaxRadiusKM:    200,
		LanguageCaseFallbacks: map[models.LanguageCasePair]models.LanguageCasePair{
			{Lang: consts.LangUK, Case: consts.CaseGenitive}: {Lang: consts.LangRU, Case: consts.CaseGenitive},
		},
		LanguageFallbacks: map[models.Lang]models.Lang{
			consts.LangTR: consts.LangEN,
			consts.LangUK: consts.LangRU,
			consts.LangDE: consts.LangEN,
			consts.LangKK: consts.LangRU,
			consts.LangBE: consts.LangRU,
		},
		DebugLogResponseJSON: false,
		GeobaseConfig:        &wizardgeobase.DefaultConfig,
		Probes:               &probes.DefaultHTTPConfig,
		FeatureFlag:          &featureflag.DefaultConfig,
	}
	helpers.PanicIfNotNil(loader.Load(context.Background(), &settings))

	if settings.YandexEnvironmentType == "production" {
		settings.FrontendURLs = getProductionFrontendURLs()
		settings.TravelPortalURLs = getProductionTravelPortalURLs()
		settings.XRedirectURLs = getProductionXredirectURLs()
	}
	return &settings
}

type FrontendURLs map[string]string

func getTestingFrontendURLs() FrontendURLs {
	return FrontendURLs{
		"ru":  "https://front.avia.tst.yandex.ru",
		"ua":  "https://front.avia.tst.yandex.ua",
		"kz":  "https://front.avia.tst.yandex.kz",
		"tr":  "https://front.avia.tst.yandex.com.tr",
		"by":  "https://front.avia.tst.yandex.ru",
		"com": "https://front.avia.tst.yandex.com",
	}
}

func getTestingTravelPortalURLs() FrontendURLs {
	return FrontendURLs{
		"ru": "https://travel-test.yandex.ru",
	}
}

func getTestingXRedirectURLs() FrontendURLs {
	return FrontendURLs{
		"ru": "https://xredirect-test.yandex.ru",
	}
}

func getProductionFrontendURLs() FrontendURLs {
	return FrontendURLs{
		"ru":  "https://avia.yandex.ru",
		"ua":  "https://avia.yandex.ua",
		"kz":  "https://avia.yandex.kz",
		"tr":  "https://bilet.yandex.com.tr",
		"by":  "https://avia.yandex.ru",
		"com": "https://flights.yandex.com",
	}
}

func getProductionTravelPortalURLs() FrontendURLs {
	return FrontendURLs{
		"ru": "https://travel.yandex.ru",
	}
}

func getProductionXredirectURLs() FrontendURLs {
	return FrontendURLs{
		"ru": "https://xredirect.yandex.ru",
	}
}
