package translations

import (
	"strings"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/tanker"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/helpers"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/repositories"
)

type FlightTranslator struct {
	timeTranslator *TimeTranslator

	tanker                    tanker.ITranslator
	translatedTitleRepository repositories.TranslatedTitle
	logger                    log.Logger
}

func NewFlightTranslator(
	tanker tanker.ITranslator,
	translatedTitleRepository repositories.TranslatedTitle,
	logger log.Logger,
	timeTranslator *TimeTranslator,
) *FlightTranslator {
	return &FlightTranslator{
		timeTranslator:            timeTranslator,
		tanker:                    tanker,
		translatedTitleRepository: translatedTitleRepository,
		logger:                    logger,
	}
}

func (translator *FlightTranslator) LocContentText(
	flightNumber string, lang models.Lang, company *models.Company,
	fromPoint, toPoint models.Point,
) (string, error) {
	placeholders, err := translator.getTitlePlaceholders(flightNumber, lang, company, fromPoint, toPoint)
	if err != nil {
		return "", err
	}

	return translator.tanker.GetTemplatedTranslation(
		"flight_wizard_content_text", lang,
		placeholders.getContentTextPlaceholder(),
	)
}

func (translator *FlightTranslator) LocTitle(
	flightNumber string, lang models.Lang, company *models.Company,
	fromSettlement, toSettlement models.Point,
) (string, error) {
	placeholders, err := translator.getTitlePlaceholders(flightNumber, lang, company, fromSettlement, toSettlement)
	if err != nil {
		return "", err
	}

	return translator.tanker.GetTemplatedTranslation(
		"Рейс {{.flight_number}} {{.from_title}} – {{.to_title}} авиакомпании {{.airline}}",
		lang, placeholders.getTitlePlaceholder(),
	)
}

func (translator *FlightTranslator) getTitlePlaceholders(
	flightNumber string, lang models.Lang, company *models.Company,
	fromPoint, toPoint models.Point,
) (titlePlaceholder, error) {
	if fromPoint == nil {
		return titlePlaceholder{}, domain.NewWizardError("couldn't parse fromPoint", domain.BadPoint)
	}
	if toPoint == nil {
		return titlePlaceholder{}, domain.NewWizardError("couldn't parse toPoint", domain.BadPoint)
	}
	companyTitle, err := translator.translatedTitleRepository.GetOldTitleTranslation(company, lang)
	if err != nil {
		return titlePlaceholder{}, err
	}

	fromTitle, err := translator.translatedTitleRepository.GetTitleTranslation(
		fromPoint.GetTitleID(), lang, consts.CaseNominative)
	if err != nil {
		return titlePlaceholder{}, err
	}

	toTitle, err := translator.translatedTitleRepository.GetTitleTranslation(
		toPoint.GetTitleID(), lang, consts.CaseNominative)
	if err != nil {
		return titlePlaceholder{}, err
	}

	return titlePlaceholder{
		flightNumber: flightNumber,
		fromTitle:    fromTitle,
		toTitle:      toTitle,
		airline:      companyTitle,
	}, nil
}

func (translator *FlightTranslator) LocLastUpdate(delta time.Duration, lang models.Lang) (string, error) {
	if delta.Seconds() < 60 {
		return translator.locRecentUpdate(lang)
	}
	locDelta := translator.LocDuration(helpers.RoundDuration(delta), lang)
	return translator.tanker.GetTemplatedTranslation(
		"Данные обновлены {{.timedelta}} назад",
		lang,
		GetPlaceholder(Pair{"timedelta", locDelta}),
	)
}

func (translator *FlightTranslator) LocSeeMore(lang models.Lang) (string, error) {
	return translator.tanker.GetTranslation("Подробнее", lang)
}

func (translator *FlightTranslator) locRecentUpdate(lang models.Lang) (string, error) {
	return translator.tanker.GetTranslation("Данные обновлены только что", lang)
}

func (translator *FlightTranslator) LocDuration(delta time.Duration, lang models.Lang) string {
	days, hours, minutes, _ := helpers.SplitDuration(delta)

	result := make([]string, 0)
	if days > 0 {
		result = append(result, suppress(translator.timeTranslator.LocDays(days, lang)))
	}
	if hours > 0 {
		result = append(result, suppress(translator.timeTranslator.LocHours(hours, lang)))
	}
	if minutes > 0 || len(result) == 0 {
		result = append(result, suppress(translator.timeTranslator.LocMinutes(minutes, lang)))
	}

	return strings.Join(result, " ")
}

func (translator *FlightTranslator) LocFlightStatus(
	status string,
	departureIsKnown,
	arrivalIsKnown bool,
	language models.Lang,
) (string, error) {
	if language == consts.LangRU && status == "unknown" {
		if !departureIsKnown && !arrivalIsKnown {
			return "Нет онлайн-информации об этом рейсе", nil
		} else if !departureIsKnown {
			return "Нет данных для аэропорта вылета", nil
		} else if !arrivalIsKnown {
			return "Нет данных для аэропорта прилёта", nil
		}
	}
	statusKey := statusToText[status]
	if statusKey == "" {
		statusKey = "unknown"
	}

	return translator.tanker.GetTranslation(statusKey, language)
}

func (translator *FlightTranslator) LocFlightStatusDiverted(
	station *models.Station,
	code string,
	language models.Lang,
) (string, error) {
	fallback := func() (string, error) {
		return translator.LocDiverted(language)
	}

	title, err := translator.translatedTitleRepository.GetTitleTranslation(
		station.NewLTitleID, language, consts.CaseNominative)
	if err != nil {
		return fallback()
	}

	if title == "" || len(title) >= 20 {
		title = code
	}

	preposition := station.TitleRuPrepositionVVoNa
	if preposition == "" {
		preposition = "в"
	}

	text, err := translator.tanker.GetTemplatedTranslation(
		"Перенаправлен {{.v_vo_na_preposition}} {{.airport}}",
		language,
		GetPlaceholder(
			Pair{"v_vo_na_preposition", preposition},
			Pair{"airport", title},
		),
	)
	if err != nil {
		return fallback()
	}
	return text, nil
}

func (translator *FlightTranslator) LocDiverted(language models.Lang) (string, error) {
	return translator.tanker.GetTranslation("Перенаправлен", language)
}

var statusToText = map[string]string{
	"unknown":   "Информация о статусе недоступна",
	"cancelled": "Отменён",
	"arrived":   "Прилетел",
	"delayed":   "Задержан",
	"diverted":  "Перенаправлен",
	"early":     "Вылет раньше",
	"on_time":   "По расписанию",
}
