package translations

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/tanker"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/lib/consts"
)

type PointToPointTranslator struct {
	tanker                 tanker.ITranslator
	logger                 log.Logger
	pointDeclinatorFactory *PointDeclinatorFactory
}

func NewPointToPointTranslator(
	tanker tanker.ITranslator,
	logger log.Logger,
	pointDeclinatorFactory *PointDeclinatorFactory) *PointToPointTranslator {
	return &PointToPointTranslator{
		tanker:                 tanker,
		logger:                 logger,
		pointDeclinatorFactory: pointDeclinatorFactory,
	}
}

func (translator *PointToPointTranslator) LocVariantType(variantType models.VariantType, lang models.Lang) *string {
	switch variantType {
	case models.VariantTypeAviaCompany:
		translation := suppress(translator.tanker.GetTranslation("Продаёт авиакомпания", lang))
		return &translation
	case models.VariantTypeTopFlight:
		translation := suppress(translator.tanker.GetTranslation("Популярный рейс", lang))
		return &translation
	case models.VariantTypeMinPrice:
		translation := suppress(translator.tanker.GetTranslation("Низкая цена", lang))
		return &translation
	case models.VariantTypePromo:
		translation := suppress(translator.tanker.GetTranslation("Спеццена", lang))
		return &translation
	}
	return nil
}

func (translator *PointToPointTranslator) LocBaggageInfo(baggageType models.BaggageType, lang models.Lang) string {
	if baggageType == models.BaggageTypeHybrid {
		return suppress(translator.tanker.GetTranslation("Возможна оплата багажа", lang))
	}
	if baggageType == models.BaggageTypeLowcost {
		return suppress(translator.tanker.GetTranslation("Платный багаж", lang))
	}
	return ""
}

func (translator *PointToPointTranslator) LocTitleWithCompany(
	companyTitle string,
	settlementFrom, settlementTo *models.Settlement,
	lang models.Lang,
) string {
	preposition := settlementTo.TitleRuPrepositionVVoNa
	if len(preposition) == 0 {
		preposition = "в"
	}
	return suppress(translator.tanker.GetTemplatedTranslationWithCase(
		"title_point_point airline",
		lang,
		GetPlaceholder(
			Pair{"company_title", companyTitle},
			Pair{"from_title", settlementFrom},
			Pair{"to_title", settlementTo},
			Pair{"rus_v_vo_na_preposition", preposition},
		),
		translator.pointDeclinatorFactory.Get(lang),
	))
}

func (translator *PointToPointTranslator) LocTitle(
	settlementFromTitle, settlementToTitle string,
	preposition string,
	lang models.Lang,
) string {
	if len(preposition) == 0 {
		preposition = "в"
	}
	return suppress(translator.tanker.GetTemplatedTranslation(
		"p2p_dynamic_title",
		lang,
		GetPlaceholder(
			Pair{"from_title", settlementFromTitle},
			Pair{"to_title", settlementToTitle},
			Pair{"rus_v_vo_na_preposition", preposition},
		),
	))
}

func (translator *PointToPointTranslator) LocGreenurl2WithCompany(
	companyTitle string,
	settlementFrom, settlementTo *models.Settlement,
	lang models.Lang,
) string {
	preposition := settlementTo.TitleRuPrepositionVVoNa
	if len(preposition) == 0 {
		preposition = "в"
	}
	return suppress(translator.tanker.GetTemplatedTranslationWithCase(
		"greenurl_2_point_point_airline",
		lang,
		GetPlaceholder(
			Pair{"company_title", companyTitle},
			Pair{"from_title", settlementFrom},
			Pair{"to_title", settlementTo},
			Pair{"rus_v_vo_na_preposition", preposition},
		),
		translator.pointDeclinatorFactory.Get(lang),
	))
}

func (translator *PointToPointTranslator) LocGreenurl2(
	settlementFrom, settlementTo *models.Settlement,
	lang models.Lang,
) string {
	preposition := settlementTo.TitleRuPrepositionVVoNa
	if len(preposition) == 0 {
		preposition = "в"
	}
	return suppress(translator.tanker.GetTemplatedTranslationWithCase(
		"Билеты на рейсы из {{case .city_from \"genitive\"}} {{.rus_v_vo_na_preposition}} {{case .city_to \"accusative\"}}",
		lang,
		GetPlaceholder(
			Pair{"city_from", settlementFrom},
			Pair{"city_to", settlementTo},
			Pair{"rus_v_vo_na_preposition", preposition},
		),
		translator.pointDeclinatorFactory.Get(lang),
	))
}

func (translator *PointToPointTranslator) LocTitleDescription(
	settlementFromTitle, settlementToTitle string,
	lang models.Lang,
) string {
	return suppress(translator.tanker.GetTemplatedTranslation(
		"point_point_title_description",
		lang,
		GetPlaceholder(
			Pair{"from_title", settlementFromTitle},
			Pair{"to_title", settlementToTitle},
		),
	))
}

func (translator *PointToPointTranslator) LocServiceName(
	lang models.Lang,
	usePortal consts.TravelPortalUsage,
) string {
	if usePortal {
		return suppress(translator.tanker.GetTranslation("greenurl_1_avia-tavel", lang))
	}
	return suppress(translator.tanker.GetTranslation("Яндекс.Авиабилеты", lang))
}
