package translations

import (
	"fmt"
	"time"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/avia/wizard/pkg/tanker"
	"a.yandex-team.ru/travel/avia/wizard/pkg/wizard/domain/models"
)

type TimeTranslator struct {
	tanker tanker.ITranslator
	logger log.Logger
}

func NewTimeTranslator(
	tanker tanker.ITranslator,
	logger log.Logger,
) *TimeTranslator {
	return &TimeTranslator{
		tanker: tanker,
		logger: logger,
	}
}

func (translator *TimeTranslator) LocDays(days int, lang models.Lang) (string, error) {
	return translator.tanker.GetTemplatedPluralizedTranslation(
		"{{.days}} д",
		lang,
		days,
		GetPlaceholder(Pair{"days", days}),
	)
}

func (translator *TimeTranslator) LocHours(hours int, lang models.Lang) (string, error) {
	return translator.tanker.GetTemplatedTranslation(
		"{{.hours}} ч",
		lang,
		GetPlaceholder(Pair{"hours", hours}),
	)
}

func (translator *TimeTranslator) LocMinutes(minutes int, lang models.Lang) (string, error) {
	return translator.tanker.GetTemplatedTranslation(
		"{{.minutes}} мин",
		lang,
		GetPlaceholder(Pair{"minutes", minutes}),
	)
}

func (translator *TimeTranslator) LocMonthDayWeekday(date time.Time, lang models.Lang) (string, error) {
	monthDay, err := translator.LocMonthDay(date, lang)
	if err != nil {
		return "", err
	}

	weekday, err := translator.LocWeekday(date.Weekday(), lang)
	if err != nil {
		return "", err
	}

	return fmt.Sprintf("%s, %s", monthDay, weekday), nil
}

func (translator *TimeTranslator) LocMonthDay(date time.Time, lang models.Lang) (string, error) {
	month, err := translator.LocMonth(date.Month(), lang)
	if err != nil {
		return "", err
	}
	return fmt.Sprintf("%d %s", date.Day(), month), nil
}

func (translator *TimeTranslator) LocMonth(month time.Month, lang models.Lang) (string, error) {
	return translator.tanker.GetTranslation(monthKeys[month], lang)
}

func (translator *TimeTranslator) LocWeekday(weekday time.Weekday, lang models.Lang) (string, error) {
	return translator.tanker.GetTranslation(weekdayKeys[weekday], lang)
}

var monthKeys = map[time.Month]string{
	time.January:   "янв",
	time.February:  "фев",
	time.March:     "мар",
	time.April:     "апр",
	time.May:       "мая",
	time.June:      "июн",
	time.July:      "июл",
	time.August:    "авг",
	time.September: "сен",
	time.October:   "окт",
	time.November:  "ноя",
	time.December:  "дек",
}

var weekdayKeys = map[time.Weekday]string{
	time.Monday:    "пн",
	time.Tuesday:   "вт",
	time.Wednesday: "ср",
	time.Thursday:  "чт",
	time.Friday:    "пт",
	time.Saturday:  "сб",
	time.Sunday:    "вс",
}
