import argparse
import logging
from itertools import chain

from travel.avia.yeah.app import create_app
from travel.avia.yeah.models.heater_config import HeaterConfig, HeaterConfig42, CustomHeaterConfig
from travel.avia.yeah.services.ticket_daemon_api.client import QueryParams
from travel.avia.yeah.tasks.heater import warming_up, NATIONAL_VERSION_DEFAULT_LANGS

logger = logging.getLogger(__name__)


def _fill_default_queue():
    logger.info('Prepare heater tasks')
    app = create_app()
    with app.app_context():
        heater_configs = HeaterConfig.query.all()
    _fill_queue(iter(heater_configs), custom_config_to_query_params)


def _fill_42_config_queue():
    logger.info('Prepare 42 config tasks')
    app = create_app()
    with app.app_context():
        heater_configs_42 = HeaterConfig42.query.filter(HeaterConfig42.format_type.in_(['42'])).all()
    _fill_queue(chain.from_iterable(heater_configs_42), custom_config_to_query_params)


def _fill_many_heater_configs_queue():
    logger.info('Prepare custom config tasks')
    app = create_app()
    with app.app_context():
        heater_configs = CustomHeaterConfig.query.filter(CustomHeaterConfig.format_type.in_(['many_heater_configs'])).all()
    _fill_queue(chain.from_iterable(heater_configs), custom_config_to_query_params)


def _fill_queue(config_generator, map_to_query_params):
    idx = 0
    for config in config_generator:
        query_params = map_to_query_params(config)
        warming_up.apply_async(args=(query_params.to_dict(),))
        idx += 1
    logger.info('Tasks sent. count=%d', idx)


def custom_config_to_query_params(sub_config):
    return QueryParams(
        from_key=sub_config['code_from'],
        to_key=sub_config['code_to'],
        date_forward=sub_config['forward_date'],
        date_backward=sub_config.get('backward_date'),
        national_version=sub_config['national_version'],
        lang=NATIONAL_VERSION_DEFAULT_LANGS.get(sub_config['national_version'], 'ru'),
        partner_codes=(),
        adults=sub_config['adults_count'],
        children=sub_config['children_count'],
        infants=sub_config['infants_count'],
        service=sub_config['service'],
    )


def heater_config_to_query_params(heater_config):
    return QueryParams(
        from_key=heater_config.code_from,
        to_key=heater_config.code_to,
        date_forward=heater_config.forward_date,
        date_backward=heater_config.backward_date,
        national_version=heater_config.national_version,
        lang=NATIONAL_VERSION_DEFAULT_LANGS.get(heater_config.national_version, 'ru'),
        partner_codes=(),
        adults=heater_config.adults_count,
        children=heater_config.children_count,
        infants=heater_config.infants_count,
        service='yeah',
    )


if __name__ == '__main__':
    parser = argparse.ArgumentParser()

    parser.add_argument('--heater', help='Heater name (default, 42, custom_config)')

    args = parser.parse_args()

    if not args.heater:
        parser.error('Empty heater name')

    if args.heater == 'default':
        _fill_default_queue()
    elif args.heater == '42':
        _fill_42_config_queue()
    elif args.heater == 'many_heater_configs':
        _fill_many_heater_configs_queue()
    else:
        parser.error('Unknown heater')
