from datetime import datetime, timedelta

from sqlalchemy import (
    Date, DateTime, SmallInteger, Column, Integer,
    BigInteger, JSON, UniqueConstraint, String
)

from travel.avia.yeah.database import db
from travel.avia.yeah.utils import time_until_end_of_day

__all__ = ('HeaterConfig', 'CustomHeaterConfig', 'HeaterConfig42')


class CustomHeaterConfig(db.Model):
    __tablename__ = 'custom_heater_config'

    name = Column(String, primary_key=True)
    format_type = Column(String, nullable=False)
    data = Column(JSON, nullable=False)
    created_at = Column(DateTime, nullable=False, default=datetime.now)
    updated_at = Column(DateTime, nullable=False, default=datetime.now, onupdate=datetime.now)


class HeaterConfig42(CustomHeaterConfig):
    def gen_configs(self):
        fwd_start = datetime.now().date()
        hours_to_tomorrow = time_until_end_of_day().total_seconds() / 60 / 60
        if hours_to_tomorrow < 6:
            fwd_start += timedelta(days=1)

        for config_42 in self.data:
            service = 'yeah_{}_{}'.format(self.name, config_42['experiment'])
            for delta in range(config_42['min_forward_delta'], config_42['max_forward_delta'] + 1):
                fwd_date_delta = timedelta(days=delta)
                config = {
                    'national_version': 'ru',
                    'adults_count': 1,
                    'children_count': 0,
                    'infants_count': 0,
                    'service': service,
                }

                config.update(config_42)
                config['forward_date'] = fwd_start + fwd_date_delta
                config['backward_date'] = None
                if config_42['travel_time'] > 0:
                    config['backward_date'] = config['forward_date'] + timedelta(
                        days=config_42['travel_time'])

                yield config


class ManyHeaterConfigs(CustomHeaterConfig):
    def gen_configs(self):
        for sub_config in self.data:
            service = 'yeah_{}_{}'.format(self.name, sub_config.get('experiment', 'no_experiment'))
            sub_config['service'] = service
            yield sub_config


class HeaterConfig(db.Model):
    __tablename__ = 'heater_config'

    pk = Column(BigInteger, primary_key=True, autoincrement=True)
    code_from = Column(String, nullable=False)  # 'c10429'
    code_to = Column(String, nullable=False)  # 'c10440'
    experiment = Column(String, nullable=False)  # 'one_day'
    req_count = Column(Integer, nullable=False)  # 1
    travel_time = Column(Integer, nullable=False)  # 0
    created_at = Column(DateTime, nullable=False, default=datetime.now)
    updated_at = Column(DateTime, nullable=False, default=datetime.now, onupdate=datetime.now)

    adults_count = Column(SmallInteger, nullable=False, default=1)
    children_count = Column(SmallInteger, nullable=False, default=0)
    infants_count = Column(SmallInteger, nullable=False, default=0)
    forward_date = Column(Date, nullable=False)
    backward_date = Column(Date, nullable=True)
    national_version = Column(String, nullable=False)

    __table_args__ = (
        UniqueConstraint(
            'national_version',
            'adults_count',
            'children_count',
            'infants_count',
            'code_from',
            'code_to',
            'forward_date',
            'backward_date',
            name='heater_config_unique',
        ),
    )

    @property
    def qkey(self):
        return (
            '{self.code_from}_{self.code_to}_{self.forward_date}_{self.backward_date}_'
            '{self.adults_count}_{self.children_count}_{self.infants_count}_{self.national_version}'
        ).format(self=self)

    def __repr__(self):
        return '<HeaterConfig qkey={self.qkey} experiment={self.experiment}'.format(self=self)
