import logging
from abc import ABCMeta, abstractmethod
from collections import namedtuple
from datetime import datetime

from requests import Session

from travel.avia.yeah import settings

logger = logging.getLogger(__name__)


JsendResponse = namedtuple('JsendResponse', ('status', 'data'))


class QueryParams(namedtuple('QueryParams', (
    'from_key', 'to_key', 'date_forward', 'date_backward',
    'national_version', 'lang', 'partner_codes',
    'adults', 'children', 'infants', 'service',
))):
    def to_dict(self):
        return self._asdict()

    @classmethod
    def from_dict(cls, query_params):
        params = query_params.copy()
        params['date_forward'] = datetime.strptime(params['date_forward'], '%Y-%m-%dT%H:%M:%S').date()
        if params['date_backward']:
            params['date_backward'] = datetime.strptime(
                params['date_backward'], '%Y-%m-%dT%H:%M:%S').date()
        return cls(**params)


class ITicketDaemonClient(object):
    __metaclass__ = ABCMeta

    @abstractmethod
    def init_search(
        self,
        point_from, point_to,
        date_forward, date_backward,
        national_version, partner_codes, lang,
        adults=0, children=0, infants=0,
        timeout=None,
    ):
        """Инициализировать поиск, вернуть qid"""

    @abstractmethod
    def results_meta(
        self, from_key, to_key, date_forward, date_backward,
        national_version,
        lang, adults=0, children=0, infants=0, service='yeah',
        timeout=None,
    ):
        """
            Вернуть мета информацию результатов по запросу
            В мете сожержится время создания и протухания вариантов
             + qid с которым был проинициализирован запрос
        """


class TicketDaemonClient(ITicketDaemonClient):
    def __init__(self, url):
        self._session = Session()
        self._url = url

    def init_search(
        self, from_key, to_key, date_forward, date_backward, national_version,
        lang, adults=0, children=0, infants=0, service='yeah', partner_codes=(),
        timeout=None,
    ):
        params = {
            'point_from': from_key,
            'point_to': to_key,
            'date_forward': date_forward.strftime('%Y-%m-%d'),
            'date_backward': (
                date_backward.strftime('%Y-%m-%d') if date_backward else None
            ),
            'national': national_version,
            'lang': lang,
            'service': service,
            'adults': adults,
            'children': children,
            'infants': infants,
        }
        if partner_codes:
            params['partner_codes'] = ','.join(sorted(set(partner_codes)))

        return self._jsend(
            'init_search',
            params=params,
            timeout=timeout,
        )

    def results_meta(
        self, from_key, to_key, date_forward, date_backward, national_version,
        lang, adults=0, children=0, infants=0, service='yeah', timeout=None,
    ):
        params = {
            'point_from': from_key,
            'point_to': to_key,
            'date_forward': date_forward.strftime('%Y-%m-%d'),
            'date_backward': (
                date_backward.strftime('%Y-%m-%d') if date_backward else None
            ),
            'national': national_version,
            'lang': lang,
            'service': service,
            'adults': adults,
            'children': children,
            'infants': infants,
        }
        return self._jsend(
            'v3/results_meta',
            params=params,
            timeout=timeout,
        )

    def _jsend(self, method, params, timeout=None):
        uri = '{}/jsendapi/{}'.format(self._url, method)
        logger.info('Sending request to %s.' % uri)
        r = self._session.get(
            uri,
            params=params,
            timeout=timeout,
        )
        r.raise_for_status()
        return JsendResponse(**r.json())


td_api_client = TicketDaemonClient(settings.TICKET_DAEMON_URL)
