from datetime import datetime

from flask import request, jsonify, Blueprint, abort
import marshmallow as ma

from travel.avia.yeah.app import db
from travel.avia.yeah.models.heater_config import HeaterConfig, CustomHeaterConfig


heater_config_bp = Blueprint('heater_config_blueprint', __name__)


class HeaterConfigSchema(ma.Schema):
    class Meta:
        fields = (
            'pk',
            'code_from',
            'code_to',
            'experiment',
            'req_count',
            'travel_time',
            'created_at',
            'updated_at',
            'adults_count',
            'children_count',
            'infants_count',
            'forward_date',
            'backward_date',
            'national_version',
        )


class CustomHeaterConfigSchema(ma.Schema):
    class Meta:
        fields = (
            'name',
            'format_type',
            'data',
            'created_at',
            'updated_at',
        )


class HeaterConfig42Schema(ma.Schema):
    req_count = ma.fields.Integer()
    travel_time = ma.fields.Integer()

    class Meta:
        fields = (
            'code_from',
            'code_to',
            'experiment',
            'national_version',
            'req_count',
            'travel_time',
            'min_forward_delta',
            'max_forward_delta',
        )


heater_config_schema = HeaterConfigSchema()
heater_configs_schema = HeaterConfigSchema(many=True)

heater_config_42_schema = HeaterConfig42Schema()
heater_configs_42_schema = HeaterConfig42Schema(many=True)


@heater_config_bp.route('/heater_config', methods=['POST'])
def add_heater_config():
    """Create new HeaterConfig endpoint"""
    ma_result = heater_config_schema.load(request.form)
    if ma_result.errors:
        pass  # todo 500
    data = ma_result.data

    new_heater_config = HeaterConfig(
        **data,
        created_at=datetime.now(),
        updated_at=datetime.now(),
    )

    db.session.add(new_heater_config)
    db.session.commit()

    result = heater_config_schema.dump(new_heater_config)
    return jsonify(result)


@heater_config_bp.route('/heater_config', methods=['GET'])
def get_heater_config():
    """Endpoint to show all HeaterConfigs"""
    all_heater_configs = HeaterConfig.query.all()
    result = heater_configs_schema.dump(all_heater_configs)
    return jsonify(result)


@heater_config_bp.route('/heater_config/<id>', methods=['GET'])
def heater_config_detail(id):
    """Endpoint to get HeaterConfig detail by id"""
    heater_config = HeaterConfig.query.get(id)
    if not heater_config:
        return abort(404)

    result = heater_config_schema.dump(heater_config)
    return jsonify(result)


@heater_config_bp.route('/heater_config/<id>', methods=['PUT'])
def heater_config_update(id):
    """Endpoint to update HeaterConfig"""
    heater_config = HeaterConfig.query.get(id)
    if not heater_config:
        return abort(404)

    ma_result = heater_config_schema.load(request.form)

    if ma_result.errors:
        pass  # todo 500

    data = ma_result.data
    if data:
        for k, v in data.items():
            setattr(heater_config, k, v)
        heater_config.updated_at = datetime.now()
        db.session.add(heater_config)
        db.session.commit()

    result = heater_config_schema.dump(heater_config)
    return jsonify(result)


@heater_config_bp.route('/heater_config/<id>', methods=['DELETE'])
def heater_config_delete(id):
    """Endpoint to delete HeaterConfig"""
    heater_config = HeaterConfig.query.get(id)
    if not heater_config:
        return abort(404)

    db.session.delete(heater_config)
    db.session.commit()

    result = heater_config_schema.dump(heater_config)
    return jsonify(result)


# Custom configs

@heater_config_bp.route('/heater_config/custom/<name>', methods=['POST'])
def add_custom_heater_config(name):
    """Create new CustomHeaterConfig endpoint with specified format.
    Format 42 is used by default:
    curl -X POST 'http://localhost:80/heater_config/custom/config_name' \
    -d '[{"travel_time": 0, "national_version": "ru', "code_from": "c213', "experiment": "tails', "req_count": 1, \
    "code_to": "c100', "min_forward_delta": 40, "max_forward_delta": 40}, {"travel_time": 7, "national_version": "ru', \
    "code_from": "c213', "experiment": "tails', "req_count": 1, "code_to": "c2', "min_forward_delta": 40, \
    "max_forward_delta": 40}]' -H "Content-Type: application/json"
    """
    data = request.json

    heater_config = CustomHeaterConfig.query.get(name)
    if not heater_config:
        heater_config = CustomHeaterConfig()
    heater_config.name = name
    heater_config.data = data

    heater_config.format_type = request.args.get('format', '42')

    db.session.add(heater_config)
    db.session.commit()

    result = CustomHeaterConfigSchema().dump(heater_config)
    return jsonify(result)


@heater_config_bp.route('/heater_config/custom', methods=['GET'])
def get_custom_heater_config():
    """Endpoint to show all CustomHeaterConfig"""
    all_heater_configs = CustomHeaterConfig.query.all()
    result = CustomHeaterConfigSchema(many=True).dump(all_heater_configs)
    return jsonify(result)


@heater_config_bp.route('/heater_config/custom/<name>', methods=['GET'])
def custom_heater_config_detail(name):
    """Endpoint to get CustomHeaterConfig detail by name"""
    heater_config = CustomHeaterConfig.query.get(name)
    if not heater_config:
        return abort(404)

    result = CustomHeaterConfigSchema().dump(heater_config)
    return jsonify(result)


@heater_config_bp.route('/heater_config/custom/<name>', methods=['DELETE'])
def custom_heater_config_delete(name):
    """Endpoint to delete CustomHeaterConfig"""
    heater_config = CustomHeaterConfig.query.get(name)
    if not heater_config:
        return abort(404)

    db.session.delete(heater_config)
    db.session.commit()

    result = CustomHeaterConfigSchema().dump(heater_config)
    return jsonify(result)
