package main

import (
	"a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/travel/budapest/bitrix_sync/internal/app"
	"a.yandex-team.ru/travel/budapest/bitrix_sync/internal/meters"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/logging"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
	"context"
	"fmt"
	"os"
	"os/signal"
	"syscall"
	"time"
)

func main() {
	// setting up infrastructure

	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &app.Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		os.Exit(1)
	}

	logger, err := configureLogger()
	if err != nil {
		fmt.Println("Unable to configure logger:", err)
		ctxCancel()
		os.Exit(1)
	}

	// metrics
	rootRegistry := metrics.NewAppMetricsRegistryWithPrefix("budapest")
	appMetrics := metrics.NewAppMetrics(rootRegistry.WithPrefix("app"))
	metrics.SetGlobalAppMetrics(appMetrics)
	meters.InitGlobalMeters()
	metrics.RunPerfMetricsUpdater(rootRegistry, app.Cfg.Metrics.PerfMetricsRefreshInterval)

	go func() {
		err := metricsserver.RunMetricsHTTPServer(context.Background(), app.Cfg.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while starting metrics server", log.Error(err))
		}
	}()

	c := make(chan os.Signal)
	signal.Notify(c, os.Interrupt, syscall.SIGTERM)
	go func() {
		sign := <-c
		logger.Info("Interrupted", log.String("signal", sign.String()))
		ctxCancel()
	}()
	logger.Info("Started service")
	defer logger.Info("Stopped service")
	for {
		select {
		case <-ctx.Done():
			return
		default:
			var runContext context.Context
			if app.Cfg.Lock.LockPath != "" {
				logger.Info("Trying to acquire the lock")
				runContext, err = app.Lock(ctx, app.Cfg.Lock.LockPath, app.Cfg.Lock.LockProxy, app.Cfg.YT.Token, logger)
				if err != nil {
					meters.SetMaster(false)
					logger.Info("Unable to acquire the lock, sleeping", log.Error(err))
					app.SleepContext(ctx, 15*time.Second)
					continue
				}
				logger.Info("Acquired the lock, we are the master, proceeding")
			} else {
				runContext = ctx
			}
			meters.SetMaster(true)
			instance := app.NewActivator(logger)
			err = instance.Run(runContext)
			if err != nil {
				panic(fmt.Errorf("unable to run activator instance: %w", err))
			}
			app.SleepContext(ctx, 2*time.Second)
		}
	}
}

func configureLogger() (*arczap.Logger, error) {
	level, err := log.ParseLevel(app.Cfg.Logging.Level)
	if err != nil {
		return nil, fmt.Errorf("bad log level format: %w", err)
	}
	var loggers []*arczap.Logger
	if app.Cfg.Logging.StdOut {
		console, err := arczap.New(arczap.ConsoleConfig(level))
		if err != nil {
			return nil, fmt.Errorf("unable to configure console logger: %w", err)
		}
		loggers = append(loggers, console)
	}
	if app.Cfg.Logging.PlainFilePath != "" {
		planFileLogger, err := logging.NewFileLogger(level, app.Cfg.Logging.PlainFilePath)
		if err != nil {
			return nil, fmt.Errorf("failed to create plain file logger: %w", err)
		}
		loggers = append(loggers, planFileLogger)
	}
	if app.Cfg.Logging.StructuredFilePath != "" {
		jsonFileLogger, err := logging.NewJSONFileLogger(level, app.Cfg.Logging.StructuredFilePath)
		if err != nil {
			return nil, fmt.Errorf("failed to create json file logger: %w", err)
		}
		loggers = append(loggers, jsonFileLogger)
	}
	if len(loggers) > 0 {
		return logging.MergeLoggers(loggers...), nil
	} else {
		return nil, nil
	}
}
