package main

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix"
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix/client"
	"context"
	"fmt"
	"reflect"
	"time"
)

type deal struct {
	Hotel                    string                  `bitrix:"-" yson:"Отель"`
	ID                       string                  `bitrix:"ID"`
	Title                    string                  `bitrix:"TITLE"`
	DealStage                *bitrix.DirectoryEntree `bitrix:"STAGE_ID" yson:"Стадия сделки"`
	RoomID                   string                  `bitrix:"!" yson:"Номер"`
	PlannedCheckin           *time.Time              `bitrix:"!" yson:"Заезд"`
	ActualCheckin            *time.Time              `bitrix:"!" yson:"Регистрация заезда"`
	PlannedCheckout          *time.Time              `bitrix:"!" yson:"Выезд"`
	ActualCheckout           *time.Time              `bitrix:"!" yson:"Регистрация выезда"`
	ReturnCustomer           bool                    `bitrix:"IS_RETURN_CUSTOMER" yson:"Повторная сделка"`
	Closed                   bool                    `bitrix:"CLOSED" yson:"Сделка закрыта"`
	Sum                      float64                 `bitrix:"OPPORTUNITY" yson:"Сумма"`
	Currency                 string                  `bitrix:"CURRENCY_ID" yson:"Валюта"`
	Source                   *bitrix.DirectoryEntree `bitrix:"SOURCE_ID" yson:"Источник"`
	PrepaySum                *float64                `bitrix:"!" yson:"Сумма к предоплате"`
	PaidSum                  *float64                `bitrix:"!" yson:"Фактически оплачено"`
	PaymentSystemName        string                  `bitrix:"!" yson:"Название способа оплаты"`
	NumAdults                *int                    `bitrix:"!" yson:"Кол-во взрослых"`
	NumChildren              *int                    `bitrix:"!" yson:"Кол-во детей"`
	Days                     *int                    `bitrix:"!" yson:"Кол-во дней проживания"`
	Nights                   *int                    `bitrix:"!" yson:"Кол-во ночей проживания"`
	RoomTypeName             string                  `bitrix:"!" yson:"Название категории номера"`
	PlacementName            string                  `bitrix:"!" yson:"Способ размещения"`
	RatePlanName             string                  `bitrix:"!" yson:"Название тарифа"`
	CustomerComment          string                  `bitrix:"!" yson:"Комментарий гостя"`
	TrafficSource            string                  `bitrix:"!" yson:"Источник трафика"`
	TrafficType              string                  `bitrix:"!" yson:"Тип трафика"`
	MarketingCampaign        string                  `bitrix:"!" yson:"Маркетинговая кампания"`
	DistributionSubscription bool                    `bitrix:"!" yson:"Согласие на получение рассылки"`
	ChannelName              string                  `bitrix:"!" yson:"OTA источник бронирования"`
	BookingNumber            string                  `bitrix:"!" yson:"№ бронирования (без идентификатора номера)"`
	CreatedAt                *time.Time              `bitrix:"DATE_CREATE" yson:"Дата создания"`
	ContactID                string                  `bitrix:"CONTACT_ID" yson:"-"`
	Contact                  *Contact                `bitrix:"-" yson:"Контакт"`
}

type importerClient struct {
	client    client.Client
	fields    wellKnownFields
	schema    *bitrix.HotelSchema
	directory bitrix.Directory
}

type wellKnownFields struct {
	ID         client.DealField
	CategoryID client.DealField
}

func createClient(cfg *hotelConfig) (*importerClient, error) {
	var c importerClient
	cfg.Schema.LoadDefaultFields(reflect.TypeOf(deal{}))
	if err := bitrix.LoadWellKnownFields(cfg.Schema, &c.fields); err != nil {
		return nil, fmt.Errorf("unable to load fields: %w", err)
	}
	if schema, err := bitrix.LoadMappingSchema(cfg.Schema, reflect.TypeOf(deal{})); err != nil {
		return nil, fmt.Errorf("unable to load schema: %w", err)
	} else {
		c.schema = schema
	}
	c.client = client.NewHTTPClient(cfg.Client)
	return &c, nil
}

func (c *importerClient) LoadDirectory(ctx context.Context) error {
	dir, err := bitrix.LoadDirectory(ctx, c.client)
	if err != nil {
		return fmt.Errorf("unable to load directory: %w", err)
	}
	c.directory = dir
	return nil
}

func (c *importerClient) ListDeals() *client.Pager {
	return c.client.GetPager(
		client.NewListQuery(bitrix.CRMDeals).
			EQ(c.fields.CategoryID, "0").
			SelectFields(c.schema.AllFields...), c.mapDeal)
}

func (c *importerClient) GetDeal(ctx context.Context, id string) (*deal, error) {
	if d, err := c.client.Get(ctx, bitrix.CRMDeals, id); err != nil {
		return nil, fmt.Errorf("unable to get deal: %w", err)
	} else {
		if r, err := c.mapDeal(d); err != nil {
			return nil, fmt.Errorf("unable to map deal: %w", err)
		} else {
			dd := r.(deal)
			return &dd, nil
		}
	}
}

func (c *importerClient) mapDeal(m map[string]interface{}) (interface{}, error) {
	return c.schema.MapToDeal(m, c.directory)
}
