package main

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix/client"
	"context"
	"fmt"
	"github.com/mitchellh/mapstructure"
	"strings"
)

const CRMContact = "crm.contact"

type Contact struct {
	ID         string `bitrix:"ID"`
	Name       string `bitrix:"NAME"`
	SecondName string `bitrix:"SECOND_NAME"`
	LastName   string `bitrix:"LAST_NAME"`
}

func listContactsByIDs(bxClient client.Client, ids []string) (map[string]*Contact, error) {
	var idsFilter []interface{}
	for _, id := range ids {
		idsFilter = append(idsFilter, id)
	}
	q := client.NewListQuery(CRMContact).
		SelectFields("NAME", "SECOND_NAME", "LAST_NAME").
		IN("ID", idsFilter...)
	res, err := bxClient.ListAll(context.Background(), q, mapToContact)
	if err != nil {
		return nil, err
	}
	result := map[string]*Contact{}
	for _, c := range res {
		con := c.(*Contact)
		result[con.ID] = con
	}
	return result, nil
}

func fillInContacts(bxClient client.Client, deals []*deal) error {
	var contactIDs []string
	for _, deal := range deals {
		contactIDs = append(contactIDs, deal.ContactID)
	}
	contacts, err := listContactsByIDs(bxClient, contactIDs)
	if err != nil {
		return err
	}
	for _, deal := range deals {
		contact, exists := contacts[deal.ContactID]
		if exists {
			deal.Contact = contact
		} else {
			fmt.Printf("Contact '%s' for deal '%s' was not found\n", deal.ContactID, deal.ID)
		}
	}
	return nil
}

func mapToContact(sourceMap map[string]interface{}) (interface{}, error) {
	var contact Contact
	decoder, err := mapstructure.NewDecoder(&mapstructure.DecoderConfig{
		Result:  &contact,
		TagName: "bitrix",
	})
	if err != nil {
		return nil, err
	}
	if err := decoder.Decode(sourceMap); err != nil {
		return nil, err
	}
	return &contact, nil
}

func (c Contact) MarshalText() (text []byte, err error) {
	var parts []string
	if c.Name != "" {
		parts = append(parts, c.Name)
	}
	if c.SecondName != "" {
		parts = append(parts, c.SecondName)
	}
	if c.LastName != "" {
		parts = append(parts, c.LastName)
	}
	return []byte(strings.Join(parts, " ")), nil
}
