package main

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/internal/common"
	"a.yandex-team.ru/travel/hotels/lib/go/ytstorage"
	"context"
	"fmt"
	"os"
	"sync"
	"time"
)

func main() {
	ctx, ctxCancel := context.WithCancel(context.Background())
	cfg := defaultConfig
	if err := common.SetupConfigForSandboxTool(ctx, &cfg); err != nil {
		fmt.Println(err)
		ctxCancel()
		os.Exit(1)
	}
	secrets, err := common.ParseBitrixSecrets(cfg.BitrixSecrets)
	if err != nil {
		fmt.Println(err)
		ctxCancel()
		os.Exit(1)
	}
	_ = cfg.YT.LoadTokenFromFile()
	for hotelName, cfg := range cfg.Hotels {
		secret, exists := secrets[hotelName]
		if !exists {
			fmt.Printf("secret for hotel '%s' is not passed\n", hotelName)
			ctxCancel()
			os.Exit(1)
		}
		cfg.Client.Secret = secret
	}

	wg := sync.WaitGroup{}
	var hasErrors bool
	for hotelName, hotelCfg := range cfg.Hotels {
		if c, err := createClient(hotelCfg); err != nil {
			fmt.Println(fmt.Errorf("unable to create client for hotel %s: %w", hotelName, err))
			ctxCancel()
			os.Exit(1)
		} else {
			if err := c.LoadDirectory(ctx); err != nil {
				fmt.Println(fmt.Errorf("unable to load directory for hotel %s: %w", hotelName, err))
				ctxCancel()
				os.Exit(1)
			}
			wg.Add(1)
			go func() {
				defer wg.Done()
				ch := listDealsToChan(ctx, c, hotelName)
				path := fmt.Sprintf("%s/%s/%s", cfg.ExportPath, hotelName, time.Now().Format(time.RFC3339))
				if err := ytstorage.Save(ctx, ch, path, 1, cfg.YT.Token, cfg.YT.Proxy); err != nil {
					fmt.Println(fmt.Errorf("unable to import data of hotel %s: %w", hotelName, err))
					hasErrors = true
					return
				}
				latestPath := fmt.Sprintf("%s/%s/%s", cfg.ExportPath, hotelName, "latest")
				if err := ytstorage.Link(ctx, path, latestPath, cfg.YT.Token, cfg.YT.Proxy); err != nil {
					fmt.Println(fmt.Errorf("unable to link latest data of hotel %s: %w", hotelName, err))
					hasErrors = true
					return
				}
			}()
		}
	}
	wg.Wait()
	if hasErrors {
		fmt.Println("errors encountered")
		os.Exit(1)
	}
}
