package main

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix"
	bxclient "a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix/client"
	"context"
	"fmt"
	"reflect"
	"time"
)

type deal struct {
	ID        string `bitrix:"ID"`
	Title     string `bitrix:"TITLE"`
	PromoCode string `bitrix:"!"`
}

type wellKnownFields struct {
	DealStage  bxclient.DealField
	CategoryID bxclient.DealField
	PromoCode  bxclient.DealField
	CreatedAt  bxclient.DealField
}

type wellKnownStages struct {
	CheckinToday     bxclient.DealStage
	CheckinTomorrow  bxclient.DealStage
	CheckinCompleted bxclient.DealStage
	StayInProgress   bxclient.DealStage
	CheckoutToday    bxclient.DealStage
	StayCompleted    bxclient.DealStage
	Cancelled        bxclient.DealStage
	Failed           bxclient.DealStage
}

type promoClient struct {
	bxClient        bxclient.Client
	fields          wellKnownFields
	stages          wellKnownStages
	schema          *bitrix.HotelSchema
	minCreationTime *time.Time
	stagesSlice     []interface{}
}

func buildClient(client bxclient.Client, hotelConfig *hotelConfig, promoConfig *promoConfig) (*promoClient, error) {
	result := promoClient{
		bxClient:        client,
		minCreationTime: promoConfig.MinCreationTime,
	}
	hotelConfig.Schema.LoadDefaultFields(reflect.TypeOf(deal{}))
	hotelConfig.Schema.DealFields["PromoCode"] = promoConfig.BitrixID

	if err := bitrix.LoadWellKnownStages(hotelConfig.Schema, &result.stages); err != nil {
		return nil, err
	}
	if err := bitrix.LoadWellKnownFields(hotelConfig.Schema, &result.fields); err != nil {
		return nil, err
	}
	if schema, err := bitrix.LoadMappingSchema(hotelConfig.Schema, reflect.TypeOf(deal{})); err != nil {
		return nil, err
	} else {
		result.schema = schema
	}
	v := reflect.ValueOf(result.stages)
	for i := 0; i < len(promoConfig.Stages); i++ {
		stageV := v.FieldByName(promoConfig.Stages[i])
		if stageV.IsZero() {
			return nil, fmt.Errorf("no known stage for '%s' specified for promo config", promoConfig.Stages[i])
		}
		result.stagesSlice = append(result.stagesSlice, stageV.String())
	}
	return &result, nil
}

func (c *promoClient) ListDealsWithNoPromo(ctx context.Context) ([]*deal, error) {
	query := bxclient.NewListQuery(bitrix.CRMDeals).
		IN(c.fields.DealStage, c.stagesSlice...).
		EQ(c.fields.CategoryID, "0").
		Empty(c.fields.PromoCode).
		SelectFields(c.schema.AllFields...)

	if c.minCreationTime != nil {
		query = query.GT(c.fields.CreatedAt, c.minCreationTime)
	}
	mapper := func(source map[string]interface{}) (interface{}, error) {
		return c.schema.MapToDeal(source, nil)
	}
	all, err := c.bxClient.ListAll(ctx, query, mapper)
	if err != nil {
		return nil, fmt.Errorf("unable to list deals with no promo: %w", err)
	}
	result := make([]*deal, len(all))
	for i, d := range all {
		dObj := d.(deal)
		result[i] = &dObj
	}
	return result, nil
}

func (c *promoClient) FindDealByPromo(ctx context.Context, promoCode string) (*deal, error) {
	query := bxclient.NewListQuery(bitrix.CRMDeals).
		EQ(c.fields.PromoCode, promoCode).
		EQ(c.fields.CategoryID, "0").
		SelectFields(c.schema.AllFields...)
	res, err := c.bxClient.List(ctx, query)
	if err != nil {
		return nil, fmt.Errorf("unable to find deals with set promo for deduplication: %w", err)
	}
	if res.Total > 1 {
		var deals []string
		for _, d := range res.Result {
			dealObj, _ := c.schema.MapToDeal(d, nil)
			deals = append(deals, dealObj.(deal).ID)
		}
		return nil, DuplicateDealError{
			PromoValue: promoCode,
			Deals:      deals,
		}
	}
	if res.Total == 0 {
		return nil, nil
	}
	rawDealMap := res.Result[0]
	dealObj, err := c.schema.MapToDeal(rawDealMap, nil)
	if err != nil {
		return nil, fmt.Errorf("unable to extract deal info from response: %w", err)
	}
	d := dealObj.(deal)
	return &d, nil
}

func (c *promoClient) SetPromoForDeal(ctx context.Context, dealID string, promoCode string) error {
	q := bxclient.NewUpdateQuery(bitrix.CRMDeals, dealID).Field(c.fields.PromoCode, promoCode)
	if err := c.bxClient.Update(ctx, q); err != nil {
		return fmt.Errorf("unable to update promoCode for deal %s: %w", dealID, err)
	}
	return nil
}
