package main

import (
	"a.yandex-team.ru/library/go/core/xerrors"
	bxclient "a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix/client"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"context"
	"fmt"
	"log"
	"time"
)

const selectQuery = "* FROM [%s] WHERE type='%s' AND deal_id=null AND expires_at > %d ORDER BY expires_at LIMIT %d"
const selectOutdatedQuery = "* FROM [%s] WHERE deal_id=null AND expires_at < %d"
const updateStatsQuery = "type, sum(1) as count FROM [%s] WHERE deal_id=null AND expires_at > %d GROUP BY type"
const daysInPromoWindow = 30
const daysInTMPPromoWindow = 3

type statRow struct {
	Type      string     `yson:"type"`
	Count     int        `yson:"count"`
	Timestamp *time.Time `yson:"timestamp"`
}

type availablePromoCode struct {
	Type      string  `yson:"type"`
	Code      string  `yson:"code"`
	ExpiresAt *uint64 `yson:"expires_at"`
}

type appliedPromoCode struct {
	Type      string  `yson:"type"`
	Code      string  `yson:"code"`
	DealID    *string `yson:"deal_id"`
	Hotel     *string `yson:"hotel"`
	DealTitle *string `yson:"deal_title"`
	AppliedAt *uint64 `yson:"applied_at"`
}

type codeToDelete struct {
	Type string `yson:"type"`
	Code string `yson:"code"`
}

type promoCodeMinimized struct {
	Code      string  `yson:"code"`
	ExpiresAt *uint64 `yson:"expires_at"`
}

type tool struct {
	Cfg config
}

func (t *tool) Run(ctx context.Context) error {
	clients := make(map[string]bxclient.Client, len(t.Cfg.Hotels))
	promoTypes := make(map[string]bool)
	for hotelName, cfg := range t.Cfg.Hotels {
		client := bxclient.NewHTTPClient(cfg.Client)
		clients[hotelName] = client
		for p := range cfg.Promo {
			promoTypes[p] = true
		}
	}
	log.Printf("Started")
	if err := t.deleteOutdatedPromos(ctx); err != nil {
		log.Println(fmt.Errorf("unable to delete outdated promos: %w", err))
	}
	if err := t.gatherStats(ctx, promoTypes); err != nil {
		log.Println(xerrors.Errorf("unable to gather stats: %w", err))
	}
	for pt := range promoTypes {
		if err := t.transferPromos(ctx, pt); err != nil {
			log.Println(fmt.Errorf("unable to transfer new promos: %w", err))
		}
		err := t.setPromosOfTypeWithRetry(ctx, pt, clients)
		if err != nil {
			return fmt.Errorf("error while running routine for type %s: %w", pt, err)
		}
	}
	return nil
}

func (t *tool) setPromosOfTypeWithRetry(ctx context.Context, promoType string, clients map[string]bxclient.Client) error {
	for {
		hasMore, err := t.setPromosOfType(ctx, promoType, clients)
		if err != nil {
			return err
		}
		if !hasMore {
			log.Printf("Done applying promocodes of type %s", promoType)
			return nil
		}
		log.Printf("Not all promocodes of type %s are applied, will run once more", promoType)
	}
}

func (t *tool) setPromosOfType(ctx context.Context, promoType string, clients map[string]bxclient.Client) (bool, error) {
	schemasByHotel := make(map[string]*promoClient, len(clients))
	dealsByHotel := make(map[string][]*deal, len(clients))
	var hasMore bool
	var numDeals int

	for hotelName, hotelConfig := range t.Cfg.Hotels {
		if promoConfig, exists := hotelConfig.Promo[promoType]; !exists {
			continue
		} else {
			schema, err := buildClient(clients[hotelName], hotelConfig, promoConfig)
			if err != nil {
				return false, fmt.Errorf("unable to build schema for hotel '%s' and promo type '%s': %w", hotelName, promoType, err)
			}
			schemasByHotel[hotelName] = schema

			deals, err := schema.ListDealsWithNoPromo(ctx)
			if err != nil {
				return false, fmt.Errorf("unable to list pending deals for hotel '%s' and promo type '%s': %w", hotelName, promoType, err)
			}
			dealsByHotel[hotelName] = append(dealsByHotel[hotelName], deals...)
			numDeals += len(deals)
		}
	}
	if numDeals == 0 {
		return false, nil
	}

	ytClient, err := t.getYtClient()
	if err != nil {
		return false, fmt.Errorf("unable to get yt client: %w", err)
	}
	tx, err := ytClient.BeginTabletTx(ctx, nil)
	if err != nil {
		return false, fmt.Errorf("unable to begin tx: %w", err)
	}
	promos, err := t.getPromos(ctx, tx, promoType, numDeals)
	if err != nil {
		return false, fmt.Errorf("unable to get available promos from storage: %w", err)
	}
	if len(promos) < numDeals {
		return false, fmt.Errorf("not enough promocodes of type %s", promoType)
	}

	var promoIndex int
	for hotelName, deals := range dealsByHotel {
		schema := schemasByHotel[hotelName]
		for _, deal := range deals {
			promo := promos[promoIndex]
			promoIndex++
			existing, err := schema.FindDealByPromo(ctx, promo.Code)
			if err != nil {
				return false, fmt.Errorf("unable to check promo for duplciates: %w", err)
			}
			if existing != nil {
				log.Printf("Code %s of type %s is already assigned, expecting duplicate", promo.Code, promo.Type)
				err = t.updatePromo(ctx, tx, promoType, promo.Code, hotelName, existing.ID, existing.Title)
				if err != nil {
					return false, fmt.Errorf("unable to duplicate-mark promo in storage: %w", err)
				}
				hasMore = true
			} else {
				log.Printf("Setting %s promo code to %s's deal with id %s", promoType, hotelName, deal.ID)
				err = schema.SetPromoForDeal(ctx, deal.ID, promo.Code)
				if err != nil {
					return false, fmt.Errorf("unable to apply promode to %s's deal with id %s: %w", hotelName, deal.ID, err)
				}
				err = t.updatePromo(ctx, tx, promoType, promo.Code, hotelName, deal.ID, deal.Title)
				if err != nil {
					return false, fmt.Errorf("unable to update promo in storage: %w", err)
				}
			}
		}
	}
	err = tx.Commit()
	if err != nil {
		return false, fmt.Errorf("unable to commit transaction: %w", err)
	}
	return hasMore, nil
}

func (t *tool) getPromos(ctx context.Context, tx yt.TabletTx, promoType string, n int) ([]availablePromoCode, error) {
	now := time.Now()
	var window int
	if promoType == "lavka-old-unused" {
		window = daysInTMPPromoWindow
	} else {
		window = daysInPromoWindow
	}
	minExpirationDate := now.AddDate(0, 0, window)
	ts := minExpirationDate.Unix()
	promoPath := t.Cfg.Path + "/promos"
	reader, err := tx.SelectRows(ctx, fmt.Sprintf(selectQuery, ypath.Path(promoPath), promoType, ts, n), &yt.SelectRowsOptions{})
	if err != nil {
		return nil, err
	}
	var res []availablePromoCode
	for reader.Next() {
		var row availablePromoCode
		err = reader.Scan(&row)
		if err != nil {
			return nil, err
		}
		res = append(res, row)
	}
	return res, reader.Err()
}

func (t *tool) gatherStats(ctx context.Context, promoTypeSet map[string]bool) error {
	now := time.Now()
	minExpirationDate := now.AddDate(0, 0, 2*daysInPromoWindow)
	ts := minExpirationDate.Unix()
	promoPath := t.Cfg.Path + "/promos"
	ytClient, err := t.getYtClient()
	if err != nil {
		return xerrors.Errorf("unable to get yt client: %w", err)
	}
	reader, err := ytClient.SelectRows(ctx, fmt.Sprintf(updateStatsQuery, ypath.Path(promoPath), ts), &yt.SelectRowsOptions{})
	if err != nil {
		return xerrors.Errorf("unable to query stats: %w", err)
	}
	stats := make(map[string]statRow, len(promoTypeSet))
	for pt := range promoTypeSet {
		stats[pt] = statRow{
			Type:      pt,
			Timestamp: &now,
		}
	}
	for reader.Next() {
		var row statRow
		err = reader.Scan(&row)
		if err != nil {
			return xerrors.Errorf("unable to read stat row: %w", err)
		}
		row.Timestamp = &now
		stats[row.Type] = row
	}
	var statsToWrite []statRow
	for _, s := range stats {
		log.Printf("We will have %d available promos of type %s in %d days", s.Count, s.Type, 2*daysInPromoWindow)
		statsToWrite = append(statsToWrite, s)
	}
	statsPathString := t.Cfg.Path + "/stats"
	statsPath := ypath.NewRich(statsPathString)

	if exists, _ := ytClient.NodeExists(ctx, statsPath, nil); !exists {
		if _, err := yt.CreateTable(ctx, ytClient, statsPath.Path, yt.WithInferredSchema(statsToWrite[0])); err != nil {
			return xerrors.Errorf("unable to create stat table: %w", err)
		}
	} else {
		statsPath.SetAppend()
	}
	writer, err := ytClient.WriteTable(ctx, statsPath, nil)
	if err != nil {
		return xerrors.Errorf("unable to create stat table writer row: %w", err)
	}
	for _, st := range statsToWrite {
		if err := writer.Write(st); err != nil {
			_ = writer.Rollback()
			return xerrors.Errorf("unable to create stat table writer row: %w", err)
		}
	}
	if err := writer.Commit(); err != nil {
		return xerrors.Errorf("unable to create stat table writer row: %w", err)
	}
	return nil
}

func (t *tool) deleteOutdatedPromos(ctx context.Context) error {
	now := time.Now().Unix()
	promoPath := t.Cfg.Path + "/promos"
	ytClient, err := t.getYtClient()
	if err != nil {
		return fmt.Errorf("unable to get yt client: %w", err)
	}
	tx, err := ytClient.BeginTabletTx(ctx, nil)
	if err != nil {
		return fmt.Errorf("unable to begin tx: %w", err)
	}
	reader, err := tx.SelectRows(ctx, fmt.Sprintf(selectOutdatedQuery, ypath.Path(promoPath), now), &yt.SelectRowsOptions{})
	if err != nil {
		return fmt.Errorf("unable to fetch outdated promos: %w", err)
	}
	var keysToDelete []interface{}
	for reader.Next() {
		var row availablePromoCode
		err = reader.Scan(&row)
		if err != nil {
			return fmt.Errorf("unable load outdated promo: %w", err)
		}
		keysToDelete = append(keysToDelete, codeToDelete{
			Type: row.Type,
			Code: row.Code,
		})
	}
	if len(keysToDelete) == 0 {
		log.Printf("No outdated promocodes")
		return nil
	}
	log.Printf("Will delete %d outdated promocodes", len(keysToDelete))
	err = tx.DeleteRows(ctx, ypath.Path(promoPath), keysToDelete, nil)
	if err != nil {
		return fmt.Errorf("unable to delete rows: %w", err)
	}
	err = tx.Commit()
	if err != nil {
		return fmt.Errorf("unable to commit tx: %w", err)
	}
	log.Printf("Outdated promocodes deleted")
	return nil
}

func (t *tool) updatePromo(ctx context.Context, tx yt.TabletTx, promoType string, code string, hotel string, dealID string, dealTitle string) error {
	upd := true
	now := uint64(time.Now().Unix())
	promoPath := t.Cfg.Path + "/promos"
	toInsert := []interface{}{&appliedPromoCode{
		Type:      promoType,
		Code:      code,
		Hotel:     &hotel,
		DealID:    &dealID,
		DealTitle: &dealTitle,
		AppliedAt: &now,
	}}
	return tx.InsertRows(ctx, ypath.Path(promoPath), toInsert, &yt.InsertRowsOptions{
		Update: &upd,
	})
}

func (t *tool) transferPromos(ctx context.Context, promoType string) error {
	client, err := t.getYtClient()
	if err != nil {
		return fmt.Errorf("unable to get yt client: %w", err)
	}
	sourcePath := t.Cfg.Path + "/" + promoType
	destPath := t.Cfg.Path + "/promos"
	sourceTable := ypath.Path(sourcePath)
	var rowCount int
	if err := client.GetNode(ctx, sourceTable.Attr("row_count"), &rowCount, nil); err != nil {
		return err
	}
	if rowCount == 0 {
		log.Printf("No new promocodes of type %s", promoType)
		return nil
	} else {
		log.Printf("Found %d promocodes of type %s, will transfer new to store", rowCount, promoType)

		rdr, err := client.ReadTable(ctx, sourceTable, nil)
		if err != nil {
			return err
		}
		var toInsert []interface{}
		for rdr.Next() {
			var pcm promoCodeMinimized
			if err := rdr.Scan(&pcm); err != nil {
				return err
			}
			toInsert = append(toInsert, availablePromoCode{
				Type:      promoType,
				Code:      pcm.Code,
				ExpiresAt: pcm.ExpiresAt,
			})
		}
		if err := rdr.Err(); err != nil {
			return err
		}

		upd := true
		if err := client.InsertRows(ctx, ypath.Path(destPath), toInsert, &yt.InsertRowsOptions{
			Update: &upd,
		}); err != nil {
			return err
		}
		spec := map[string]interface{}{
			"table_path": sourceTable,
		}
		opID, err := client.StartOperation(ctx, yt.OperationErase, spec, nil)
		if err != nil {
			return err
		} else {
			log.Printf("Started erase operation %s on table %s", opID.String(), sourcePath)
			return nil
		}
	}
}

func (t *tool) getYtClient() (yt.Client, error) {
	client, err := ythttp.NewClient(&yt.Config{
		Proxy: t.Cfg.YT.Proxy,
		Token: t.Cfg.YT.Token,
	})
	if err != nil {
		return nil, err
	}
	return client, nil
}
