package main

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/internal/app"
	"a.yandex-team.ru/travel/hotels/lib/go/ytstorage"
	"context"
	"fmt"
	"github.com/heetch/confita"
	"github.com/heetch/confita/backend/env"
	"github.com/spf13/cobra"
)

const DefaultYtPath = "//home/travel/budapest/activator/rooms"

func main() {
	rootCMD := &cobra.Command{
		Use: "room-manager",
	}
	addCMD := &cobra.Command{
		Use:   "add-room <bitrix-room-id> <alice-room-id> [--do-not-activate] [--do-not-apply-promo]",
		Short: "Add a room to activator checks",
		RunE:  addRoom,
	}
	addCMD.Flags().Bool("do-not-activate", false, "skip auto activation of the room on guest checkin")
	addCMD.Flags().Bool("do-not-apply-promo", false, "do not apply promo code when activating room on checkin")
	addCMD.Args = cobra.ExactArgs(2)
	removeCMD := &cobra.Command{
		Use:   "remove-room <bitrix-room-id>",
		Short: "Remove a room from activator checks",
		RunE:  removeRoom,
	}
	removeCMD.Args = cobra.ExactArgs(1)

	rootCMD.AddCommand(addCMD, removeCMD)
	_ = rootCMD.Execute()
}

func getStorage() (*app.Storage, error) {
	storageCfg := app.DefaultStorageConfig
	storageCfg.YTPath = DefaultYtPath
	ytCfg := ytstorage.DefaultConfig
	_ = ytCfg.LoadTokenFromFile()
	loader := confita.NewLoader(env.NewBackend())
	err := loader.Load(context.Background(), &storageCfg)
	if err != nil {
		return nil, err
	}
	err = loader.Load(context.Background(), &ytCfg)
	if err != nil {
		return nil, err
	}
	storage := app.NewStorage(storageCfg, ytCfg)
	return &storage, nil
}

func addRoom(cmd *cobra.Command, args []string) error {
	storage, err := getStorage()
	if err != nil {
		return err
	}
	if err := storage.Load(context.Background()); err != nil {
		return err
	}
	room := &app.Room{
		BitrixRoomID: args[0],
		A4bID:        args[1],
	}
	doNotActivate, _ := cmd.Flags().GetBool("do-not-activate")
	noAutoPromo, _ := cmd.Flags().GetBool("do-not-apply-promo")
	if doNotActivate {
		room.AutoActivate = false
		room.AutoPromoCode = false
	} else {
		room.AutoActivate = true
		if noAutoPromo {
			room.AutoPromoCode = false
		} else {
			room.AutoPromoCode = true
		}
	}
	storage.AddRoom(room)
	if err := storage.Save(context.Background()); err != nil {
		return err
	}
	fmt.Println("Room added")
	return nil
}

func removeRoom(cmd *cobra.Command, args []string) error {
	storage, err := getStorage()
	if err != nil {
		return err
	}
	if err := storage.Load(context.Background()); err != nil {
		return err
	}
	if err := storage.RemoveRoom(args[0]); err != nil {
		return err
	}
	if err := storage.Save(context.Background()); err != nil {
		return err
	}
	fmt.Println("Room removed")
	return nil
}
