package app

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/budapest/bitrix_sync/internal/meters"
	"a.yandex-team.ru/travel/budapest/bitrix_sync/internal/yameta"
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/alice4business"
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/tgalerts"
	"context"
	"fmt"
	"time"
)

type Activator struct {
	Storage
	logger       log.Logger
	bitrix       yameta.Client
	devices      *alice4business.Client
	cancels      map[string]context.CancelFunc
	alerter      *tgalerts.Alerter
	initializing bool
}

func (a *Activator) Run(ctx context.Context) error {
	var alerter *tgalerts.Alerter = nil
	if Cfg.TG.TGToken != "" {
		var err error
		alerter, err = tgalerts.NewAlerter(ctx, Cfg.TG, Cfg.YT, a.logger)
		if err != nil {
			return fmt.Errorf("unable to initialize tg alerter: %w", err)
		}
		a.alerter = alerter
		defer a.alerter.Stop()
	}
	defer func() {
		err := a.Save(context.Background()) // background, as ctx may already be cancelled
		if err != nil {
			a.logger.Warn("Unable to save storage on exit", log.Error(err))
		} else {
			a.logger.Info("Context saved on exit")
		}
	}()
	a.logger.Info("Starting activator instance")
	for {
		select {
		case <-ctx.Done():
			a.logger.Info("Stopping activator instance")
			return nil
		default:
			a.logger.Debug("Updating list of active rooms")
			err := a.updateRooms(ctx)
			meters.CountRoomUpdate(err)
			if err != nil {
				a.logger.Error("Error while updating rooms", log.Error(err))
			} else {
				a.logger.Debug("Done updating rooms")
			}
			SleepContext(ctx, Cfg.Poll.RoomListUpdateInterval)
		}
	}
}

func NewActivator(logger log.Logger) *Activator {
	return &Activator{
		Storage:      NewStorage(Cfg.Storage, Cfg.YT),
		bitrix:       yameta.NewClient(Cfg.Bitrix),
		devices:      alice4business.NewClient(Cfg.A4B),
		cancels:      make(map[string]context.CancelFunc),
		logger:       logger,
		initializing: true,
	}
}

func (a *Activator) updateRooms(ctx context.Context) error {
	rooms, err := a.loadImpl(ctx)
	if err != nil {
		return fmt.Errorf("error while loading room storage updates: %w", err)
	}
	for roomID, cancel := range a.cancels {
		if _, found := rooms[roomID]; !found {
			a.logRoomID("Комната более не проверяется, останавливаем контекст", roomID)
			cancel()
			delete(a.cancels, roomID)
		}
	}
	for roomID := range a.rooms {
		if _, found := rooms[roomID]; !found {
			a.logRoomID("Комната более не проверяется, удаляем комнату", roomID)
			delete(a.rooms, roomID)
			meters.CountRoomRemoved()
		}
	}

	for roomID, room := range rooms {
		if _, found := a.rooms[roomID]; !found {
			if !a.initializing {
				a.logRoom("Новая комната, запускаем цикл проверки", room)
				meters.CountRoomAdded()
			}
			meters.RegisterRoom(roomID)
			a.rooms[roomID] = room
			nested, cancel := context.WithCancel(ctx)
			a.cancels[roomID] = cancel
			go a.runRoom(nested, room)
		}
	}
	meters.SetRoomTotal(len(a.rooms))
	a.initializing = false
	err = a.Save(ctx)
	if err != nil {
		return fmt.Errorf("error while saving room storage: %w", err)
	}
	return nil
}

func (a *Activator) runRoom(ctx context.Context, room *Room) {
	defer a.logRoomDebug("Цикл проверки комнаты завершен", room)
	for {
		timeToSleep, err := a.checkRoom(room)
		meters.CountCheck(room.BitrixRoomID, err)
		if err != nil {
			a.logRoomErr("Ошибка при проверке комнаты", room, err)
			if SleepContext(ctx, Cfg.Poll.ErrorInterval) {
				return
			}
		}
		if SleepContext(ctx, timeToSleep) {
			return
		}
	}
}

func (a *Activator) checkRoom(room *Room) (time.Duration, error) {
	activeDeal, err := a.bitrix.GetActiveDealForRoom(room.BitrixRoomID)
	if err != nil {
		if duplicateErr, isDuplicate := err.(yameta.DuplicateDealError); isDuplicate {
			if a.alerter != nil {
				a.alerter.NonRepeatableAlert(fmt.Sprintf("Для комнаты %s найдено %d активных сделок, проверьте битрикс", duplicateErr.Room, len(duplicateErr.Deals)),
					fmt.Sprintf("duplicate_%s", duplicateErr.Room), time.Hour)
			}
		}
		return 0, fmt.Errorf("unable to get info from bitrix: %w", err)
	}
	upcomingDeal, err := a.bitrix.GetUpcomingDealForRoom(room.BitrixRoomID)
	if err != nil {
		if duplicateErr, isDuplicate := err.(yameta.DuplicateDealError); isDuplicate {
			if a.alerter != nil {
				a.alerter.NonRepeatableAlert(fmt.Sprintf("Для комнаты %s найдено %d сделок с заселением сегодня, проверьте битрикс", duplicateErr.Room, len(duplicateErr.Deals)),
					fmt.Sprintf("duplicate_%s", duplicateErr.Room), time.Hour)
			}
		}
		return 0, fmt.Errorf("unable to get info from bitrix: %w", err)
	}
	hasActiveOrUpcomingSoon := false
	if activeDeal != nil {
		hasActiveOrUpcomingSoon = true
	} else {
		if upcomingDeal != nil {
			if upcomingDeal.PlannedCheckin.Add(-2 * time.Hour).Before(time.Now()) {
				hasActiveOrUpcomingSoon = true
			}
		}
	}
	roomInA4b, err := a.devices.GetRoom(room.A4bID)
	if err != nil {
		return 0, fmt.Errorf("unable to get room info from a4b: %w", err)
	}
	status := roomInA4b.Status
	room.A4BStatus = status
	if activeDeal == nil {
		room.CurrentDeal = nil
		room.BitrixStatus = nil
	} else {
		defer func() { // defer as current deal id is used to detect deal change
			room.CurrentDeal = &activeDeal.ID
			room.BitrixStatus = &activeDeal.DealStage.ID
		}()
	}
	switch {
	case status == alice4business.RoomReset:
		if !roomInA4b.InProgress {
			if hasActiveOrUpcomingSoon {
				// неудачный сброс, скоро заезд => ничего не делаем, сообщаем в чатик
				a.reportResetFailureAndManualProcessing(room)
				return Cfg.Poll.DefaultInterval, nil
			} else {
				// неудачный сброс => Ретраим сброс независимо от состояния комнаты в Битриксе
				a.logRoomDeal("Ошибка сброса", room, activeDeal)
				a.reportResetFailure(room)
				meters.CountResetErr(room.BitrixRoomID)
				if _, err := a.devices.ResetRoom(room.A4bID); err != nil {
					return 0, fmt.Errorf("unable to reset room in a4b: %w", err)
				}
				meters.CountReset(room.BitrixRoomID, meters.Retry)
				return Cfg.Poll.OperationInProgressInterval, nil
			}
		}
		a.logRoomDeal("Ожидаем окончания сброса", room, activeDeal)
		return Cfg.Poll.OperationInProgressInterval, nil
	case activeDeal == nil && (status == alice4business.RoomActive || status == alice4business.RoomMixedStatus):
		// в битриксе нет активного заказа, устройства активны => случился чекаут, сбрасываем устройства
		if _, err := a.devices.ResetRoom(room.A4bID); err != nil {
			return 0, fmt.Errorf("unable to reset room: %w", err)
		}
		a.logRoomDeal("Инициирован сброс", room, activeDeal)
		meters.CountReset(room.BitrixRoomID, meters.Checkout)
		return Cfg.Poll.OperationInProgressInterval, nil
	case activeDeal != nil && status == alice4business.RoomInactive:
		// в битриксе есть активный заказ, устройства неактивны => случился заезд, активируем устройства
		if room.AutoActivate {
			if _, err := a.devices.ActivateRoom(room.A4bID, room.AutoPromoCode); err != nil {
				return 0, fmt.Errorf("unable to activate room in a4b: %w", err)
			}
			a.logRoomDeal("Инициирована активация", room, activeDeal)
			meters.CountActivate(room.BitrixRoomID)
			return Cfg.Poll.OperationInProgressInterval, nil
		} else {
			a.logRoomDeal("Требуется ручная активация", room, activeDeal)
			return Cfg.Poll.DefaultInterval, nil
		}
	case activeDeal != nil && status == alice4business.RoomActive:
		if room.CurrentDeal != nil && *room.CurrentDeal != activeDeal.ID {
			// в битриксе к комнате привязан другой заказ, не тот что сохранен у нас
			a.logRoomDeal("Изменение активного заказа", room, activeDeal, log.String("OldDealID", *room.CurrentDeal))
			if _, err := a.devices.ResetRoom(room.A4bID); err != nil {
				return 0, fmt.Errorf("unable to reset room in a4b: %w", err)
			}
			meters.CountReset(room.BitrixRoomID, meters.DealChange)
			return Cfg.Poll.OperationInProgressInterval, nil
		}

		if roomInA4b.PromoStatus == alice4business.RoomPromoAvailable && room.AutoPromoCode {
			a.logRoomDeal("Promo code is not applied, although it has to", room, activeDeal)
			meters.CountPromoErr(room.BitrixRoomID)
			if a.alerter != nil {
				a.alerter.NonRepeatableAlert(fmt.Sprintf("Не удалось применить промокод после активации "+
					"комнаты %s. Возможно промокоды закончились", room.BitrixRoomID),
					fmt.Sprintf("promo_na_%s", room.BitrixRoomID), time.Hour)
			}
		}
		if activeDeal.ActivationID == "" {
			// в битриксе есть активный заказ для которого нет активационной ссылки => генерируем ссылку и отправляем ее в Битрикс
			now := time.Now()
			activationID, err := a.devices.CreateRoomActivation(room.A4bID, &now, activeDeal.GetCheckout())
			if err != nil {
				return 0, fmt.Errorf("unable to generate activation link in a4b: %w", err)
			}
			link := fmt.Sprintf("%s?activationId=%s", Cfg.A4B.CustomerPage, activationID)
			link, err = shorten(link)
			if err != nil {
				return 0, fmt.Errorf("unable to shorten activation link: %w", err)
			}
			a.logRoomDeal("Генерация активационной ссылки", room, activeDeal, log.String("ActivationID", activationID))
			if err := a.bitrix.SetActivationLinkForDeal(activeDeal.ID, link); err != nil {
				return 0, fmt.Errorf("unable to set activation link in bitrix: %w", err)
			}
			meters.CountActivationLink(room.BitrixRoomID)
		} else {
			a.logRoomDealDebug("Ничего не происходит", room, activeDeal)
		}
		return Cfg.Poll.DefaultInterval, nil
	default:
		a.logRoomDealDebug("Ничего не происходит", room, activeDeal)
		return Cfg.Poll.DefaultInterval, nil
	}
}

func (a *Activator) reportResetFailure(roomOrDevice *Room) {
	alert := fmt.Sprintf("Сброс утройств в комнате %s неуспешен, пробуем еще раз.\n"+
		"Пожалуйста, проверьте, что там могло пойти не так", roomOrDevice.BitrixRoomID)
	if a.alerter != nil {
		a.alerter.Alert(alert)
	}
}

func (a *Activator) reportResetFailureAndManualProcessing(roomOrDevice *Room) {
	alert := fmt.Sprintf("Сброс утройств в комнате %s неуспешен.\n"+
		"В комнату запланирован заезд (или в ней уже живут), поэтому повторных попыток сброса не будет.\n"+
		"Пожалуйста, проверьте, что там могло пойти не так и активируйте устройства вручную, можно по одному", roomOrDevice.BitrixRoomID)
	if a.alerter != nil {
		a.alerter.NonRepeatableAlert(alert, fmt.Sprintf("manual_reset_%s", roomOrDevice.BitrixRoomID), time.Hour*24)
	}
}

func (a *Activator) logRoomDeal(message string, room *Room, deal *yameta.Deal, other ...log.Field) {
	var dealID string
	var dealTitle string
	var dealStage string
	if deal != nil {
		dealID = deal.ID
		dealTitle = deal.Title
		dealStage = deal.DealStage.Name
	}
	fields := []log.Field{log.String("roomID", room.BitrixRoomID), log.String("dealID", dealID),
		log.String("dealTitle", dealTitle), log.String("dealStage", dealStage)}
	fields = append(fields, other...)
	a.logger.Info(message, fields...)
}

func (a *Activator) logRoomDealDebug(message string, room *Room, deal *yameta.Deal, other ...log.Field) {
	var dealID string
	var dealTitle string
	var dealStage string
	if deal != nil {
		dealID = deal.ID
		dealTitle = deal.Title
		dealStage = deal.DealStage.Name
	}
	fields := []log.Field{log.String("roomID", room.BitrixRoomID), log.String("dealID", dealID),
		log.String("dealTitle", dealTitle), log.String("dealStage", dealStage)}
	fields = append(fields, other...)
	a.logger.Debug(message, fields...)
}

func (a *Activator) logRoomID(message string, roomID string) {
	a.logger.Info(message, log.String("roomID", roomID))
}

func (a *Activator) logRoom(message string, room *Room) {
	a.logger.Info(message, log.String("roomID", room.BitrixRoomID))
}

func (a *Activator) logRoomDebug(message string, room *Room) {
	a.logger.Debug(message, log.String("roomID", room.BitrixRoomID))
}

func (a *Activator) logRoomErr(message string, room *Room, err error) {
	a.logger.Error(message, log.String("roomID", room.BitrixRoomID), log.Error(err))
}

func SleepContext(ctx context.Context, delay time.Duration) bool {
	select {
	case <-ctx.Done():
		return true
	case <-time.After(delay):
		return false
	}
}
