package app

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/yt/go/ypath"
	"a.yandex-team.ru/yt/go/yt"
	"a.yandex-team.ru/yt/go/yt/ythttp"
	"context"
	"fmt"
	"time"
)

func Lock(ctx context.Context, lockPath string, lockProxy string, ytToken string, logger log.Logger) (context.Context, error) {
	client, err := ythttp.NewClient(&yt.Config{
		Proxy: lockProxy,
		Token: ytToken,
	})
	if err != nil {
		return nil, fmt.Errorf("unable to create locking client: %v", err)
	}
	txContext, txCancel := context.WithCancel(ctx)
	tx, err := client.BeginTx(txContext, nil)
	if err != nil {
		defer txCancel()
		return nil, fmt.Errorf("unable to start locking transaction: %w", err)
	}
	_, err = tx.LockNode(txContext, ypath.Path(lockPath), yt.LockExclusive, nil)

	if err != nil {
		defer txCancel()
		return nil, fmt.Errorf("unable to take a lock: %w", err)
	} else {
		go func() {
			for {
				select {
				case <-txContext.Done():
					logger.Warn("Parent context is stopped, will no longer ping tx", log.String("Tx", tx.ID().String()))
					txCancel()
					return
				default:
					err := client.PingTx(txContext, tx.ID(), nil)
					if err != nil {
						logger.Warn("Transaction is dead, cancelling context execution", log.String("Tx", tx.ID().String()), log.Error(err))
						txCancel()
						return
					} else {
						SleepContext(txContext, time.Second*15)
					}
				}
			}
		}()
		return txContext, nil
	}
}
