package app

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/alice4business"
	"a.yandex-team.ru/travel/hotels/lib/go/ytstorage"
	"context"
	"encoding/gob"
	"fmt"
	"log"
	"os"
	"reflect"
)

type Room struct {
	BitrixRoomID  string                    `yson:"bitrix_room_id"`
	A4bID         string                    `yson:"alice_room_id"`
	AutoActivate  bool                      `yson:"auto_activate"`
	AutoPromoCode bool                      `yson:"auto_promo"`
	CurrentDeal   *string                   `yson:"current_deal"`
	A4BStatus     alice4business.RoomStatus `yson:"a4b_status"`
	BitrixStatus  *string                   `yson:"bitrix_status"`
}

type Storage struct {
	config   StorageConfig
	ytConfig ytstorage.Config
	rooms    map[string]*Room
}

type ObservableStorage struct {
	Storage
	cancels map[string]context.CancelFunc
}

func (s *Storage) Load(ctx context.Context) error {
	rooms, err := s.loadImpl(ctx)
	if err != nil {
		return err
	}
	s.rooms = rooms
	return nil
}

func (s *Storage) loadImpl(ctx context.Context) (map[string]*Room, error) {

	result := map[string]*Room{}
	rooms, err := ytstorage.Load(ctx, reflect.TypeOf(Room{}), s.config.YTPath, s.ytConfig.Token, s.ytConfig.Proxy)
	if err != nil {
		log.Println(fmt.Errorf("error while reading room storage from yt: %w", err))
		if s.config.FilePath != "" {
			file, err := os.Open(s.config.FilePath)
			if err != nil {
				return nil, fmt.Errorf("unable to load storage from file as a fallback after yt failure: %w", err)
			}
			defer func() { _ = file.Close() }()
			err = gob.NewDecoder(file).Decode(&result)
			if err != nil {
				return nil, err
			}
			return result, nil
		} else {
			return nil, err
		}
	}
	for _, r := range rooms {
		room := r.(*Room)
		if room.CurrentDeal != nil && *room.CurrentDeal == "" { // backwards compatibility
			room.CurrentDeal = nil
		}
		result[room.BitrixRoomID] = room
	}
	return result, nil
}

func (s *Storage) Save(ctx context.Context) error {
	var rooms []Room
	for _, r := range s.rooms {
		rooms = append(rooms, *r)
	}
	var result error = nil
	if err := ytstorage.Save(ctx, rooms, s.config.YTPath, 1, s.ytConfig.Token, s.ytConfig.Proxy); err != nil {
		log.Println(fmt.Errorf("unable to save rooms to yt: %w", err))
		result = err
	}
	if s.config.FilePath != "" {
		file, err := os.Create(s.config.FilePath)
		if err != nil {
			log.Println(fmt.Errorf("unable to create file to save rooms to: %w", err))
			if result == nil {
				result = err
			} else {
				result = fmt.Errorf("%v, %v", result, err)
			}
			return result
		}
		defer func() { _ = file.Close() }()
		if err := gob.NewEncoder(file).Encode(s.rooms); err != nil {
			log.Println(fmt.Errorf("unable to save rooms to file: %w", err))
			if result == nil {
				result = err
			} else {
				result = fmt.Errorf("%v, %v", result, err)
			}
		}
	}
	return result
}

func (s *Storage) AddRoom(room *Room) {
	s.rooms[room.BitrixRoomID] = room
}

func (s *Storage) RemoveRoom(roomID string) error {
	if _, found := s.rooms[roomID]; found {
		delete(s.rooms, roomID)
		return nil
	} else {
		return fmt.Errorf("room not found")
	}
}

func NewStorage(config StorageConfig, ytConfig ytstorage.Config) Storage {
	return Storage{
		rooms:    make(map[string]*Room),
		config:   config,
		ytConfig: ytConfig,
	}
}
