package meters

import (
	coreMetrics "a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/travel/library/go/metrics"
)

type ResetReason string

const (
	rooms  string = "rooms"
	checks string = "checks"

	reset          string = "reset"
	activate       string = "activate"
	activationLink string = "activationLink"

	Checkout   ResetReason = "checkout"
	Retry      ResetReason = "retry"
	DealChange ResetReason = "dealChange"

	resetError string = "reset"
	promoError string = "promo"

	reason string = "reason"

	room string = "room"
)

var resetReasons = []ResetReason{Checkout, Retry, DealChange}

type roomMeters struct {
	activatesCounter       coreMetrics.Counter
	activationLinksCounter coreMetrics.Counter
	resetErrorCounter      coreMetrics.Counter
	promoErrorCounter      coreMetrics.Counter
	resetsCountersByReason map[ResetReason]coreMetrics.Counter
	okCheckCounter         coreMetrics.Counter
	errCheckCounter        coreMetrics.Counter
}

var roomMetersByRoom = map[string]roomMeters{}

func RegisterRoom(roomID string) {
	tags := map[string]string{room: roomID}
	m := roomMeters{
		activatesCounter:       metrics.GlobalAppMetrics().GetOrCreateCounter(rooms, tags, activate),
		activationLinksCounter: metrics.GlobalAppMetrics().GetOrCreateCounter(rooms, tags, activationLink),
		resetErrorCounter:      metrics.GlobalAppMetrics().GetOrCreateCounter(rooms, tags, resetError),
		promoErrorCounter:      metrics.GlobalAppMetrics().GetOrCreateCounter(rooms, tags, promoError),
		resetsCountersByReason: map[ResetReason]coreMetrics.Counter{},
	}
	for _, r := range resetReasons {
		tags := map[string]string{room: roomID, reason: string(r)}
		m.resetsCountersByReason[r] = metrics.GlobalAppMetrics().GetOrCreateCounter(rooms, tags, reset)
	}
	okTags := map[string]string{room: roomID, outcome: ok}
	errTags := map[string]string{room: roomID, outcome: err}
	m.okCheckCounter = metrics.GlobalAppMetrics().GetOrCreateCounter(rooms, okTags, checks)
	m.errCheckCounter = metrics.GlobalAppMetrics().GetOrCreateCounter(rooms, errTags, checks)
	roomMetersByRoom[roomID] = m
}

func CountReset(roomID string, r ResetReason) {
	if m, exists := roomMetersByRoom[roomID]; exists {
		if c, exists2 := m.resetsCountersByReason[r]; exists2 {
			c.Inc()
		}
	}
}

func CountActivate(roomID string) {
	if m, exists := roomMetersByRoom[roomID]; exists {
		m.activatesCounter.Inc()
	}
}

func CountPromoErr(roomID string) {
	if m, exists := roomMetersByRoom[roomID]; exists {
		m.promoErrorCounter.Inc()
	}
}

func CountResetErr(roomID string) {
	if m, exists := roomMetersByRoom[roomID]; exists {
		m.resetErrorCounter.Inc()
	}
}

func CountActivationLink(roomID string) {
	if m, exists := roomMetersByRoom[roomID]; exists {
		m.activationLinksCounter.Inc()
	}
}

func CountCheck(roomID string, err error) {
	if m, exists := roomMetersByRoom[roomID]; exists {
		if err != nil {
			m.errCheckCounter.Inc()
		} else {
			m.okCheckCounter.Inc()
		}
	}
}
