package yameta

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix/client"
	"context"
	"fmt"
)

const (
	CRMDeals   = "crm.deal"
	CRMStatus  = "crm.status"
	CRMContact = "crm.contact"
)

const (
	CheckinToday     string = "1"
	CheckinTomorrow  string = "76"
	CheckinCompleted string = "77"
	StayInProgress   string = "2"
	CheckoutToday    string = "FINAL_INVOICE"
	StayCompleted    string = "WON"
	Cancelled        string = "LOST"
)

type Client struct {
	bitrix    client.Client
	directory map[string]DirectoryEntree
}

func (c *Client) GetUpcomingDealForRoom(roomID string) (*Deal, error) {
	query := client.NewListQuery(CRMDeals).
		EQ(dealFields.RoomID.BitrixName, roomID).
		EQ(dealFields.DealStage.BitrixName, CheckinToday).
		SelectFields(allFieldsSlice...)
	res, err := c.bitrix.List(context.Background(), query)
	if err != nil {
		return nil, fmt.Errorf("unable to list active deals for room %s: %w", roomID, err)
	}
	if res.Total > 1 {
		var deals []string
		for _, d := range res.Result {
			deal, _ := c.mapToDeal(d)
			deals = append(deals, deal.ID)
		}
		return nil, DuplicateDealError{
			Room:  roomID,
			Deals: deals,
		}
	}
	if res.Total == 0 {
		return nil, nil
	}
	rawDealMap := res.Result[0]
	result, err := c.mapToDeal(rawDealMap)
	if err != nil {
		return nil, fmt.Errorf("unable to extract deal info from response: %w", err)
	}
	return result, nil
}

func (c *Client) GetActiveDealForRoom(roomID string) (*Deal, error) {
	query := client.NewListQuery(CRMDeals).
		EQ(dealFields.RoomID.BitrixName, roomID).
		IN(dealFields.DealStage.BitrixName, CheckinCompleted, StayInProgress, CheckoutToday).
		SelectFields(allFieldsSlice...)
	res, err := c.bitrix.List(context.Background(), query)
	if err != nil {
		return nil, fmt.Errorf("unable to list active deals for room %s: %w", roomID, err)
	}
	if res.Total > 1 {
		var deals []string
		for _, d := range res.Result {
			deal, _ := c.mapToDeal(d)
			deals = append(deals, deal.ID)
		}
		return nil, DuplicateDealError{
			Room:  roomID,
			Deals: deals,
		}
	}
	if res.Total == 0 {
		return nil, nil
	}
	rawDealMap := res.Result[0]
	result, err := c.mapToDeal(rawDealMap)
	if err != nil {
		return nil, fmt.Errorf("unable to extract deal info from response: %w", err)
	}
	return result, nil
}

func (c *Client) SetActivationLinkForDeal(dealID string, link string) error {
	query := client.NewUpdateQuery(CRMDeals, dealID).Field(dealFields.ActivationID.BitrixName, link)
	err := c.bitrix.Update(context.Background(), query)
	if err != nil {
		return fmt.Errorf("unable to update activation id for deal %s", dealID)
	}
	return nil
}

func (c *Client) GetDealByID(dealID string) (*Deal, error) {
	res, err := c.bitrix.Get(context.Background(), CRMDeals, dealID)
	if err != nil {
		return nil, fmt.Errorf("unable to get deal with id %s: %w", dealID, err)
	}
	return c.mapToDeal(res)
}

func (c *Client) ListDeals() *client.Pager {
	query := client.NewListQuery(CRMDeals).
		SelectFields(allFieldsSlice...)
	mapper := func(source map[string]interface{}) (interface{}, error) {
		return c.mapToDeal(source)
	}
	return c.bitrix.GetPager(query, mapper)
}

var DefaultMetaConfig = client.Config{
	RootURL: "https://yameta.bitrix24.ru/rest/281",
}

func NewClient(config client.Config) Client {
	return Client{bitrix: client.NewHTTPClient(config)}
}
