package yameta

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix/client"
	"context"
	"fmt"
	"github.com/mitchellh/mapstructure"
	"strings"
)

type Contact struct {
	ID         string `bitrix:"ID"`
	Name       string `bitrix:"NAME"`
	SecondName string `bitrix:"SECOND_NAME"`
	LastName   string `bitrix:"LAST_NAME"`
}

func (c *Client) ListContactsByIDs(ids []string) (map[string]Contact, error) {
	var idsFilter []interface{}
	for _, id := range ids {
		idsFilter = append(idsFilter, id)
	}
	q := client.NewListQuery(CRMContact).
		SelectFields("NAME", "SECOND_NAME", "LAST_NAME").
		IN("ID", idsFilter...)
	res, err := c.bitrix.List(context.Background(), q)
	if err != nil {
		return nil, err
	}
	result := map[string]Contact{}
	for _, c := range res.Result {
		con, err := mapToContact(c)
		if err != nil {
			return nil, err
		}
		result[con.ID] = *con
	}
	return result, nil
}

func (c *Client) FillInContacts(deals []*Deal) error {
	var contactIDs []string
	for _, deal := range deals {
		contactIDs = append(contactIDs, deal.ContactID)
	}
	contacts, err := c.ListContactsByIDs(contactIDs)
	if err != nil {
		return err
	}
	for _, deal := range deals {
		contact, exists := contacts[deal.ContactID]
		if exists {
			deal.Contact = &contact
		} else {
			fmt.Printf("Contant %s not found\n", deal.ContactID)
		}
	}
	return nil
}

func mapToContact(sourceMap map[string]interface{}) (*Contact, error) {
	var contact Contact
	decoder, err := mapstructure.NewDecoder(&mapstructure.DecoderConfig{
		Result:  &contact,
		TagName: "bitrix",
	})
	if err != nil {
		return nil, err
	}
	if err := decoder.Decode(sourceMap); err != nil {
		return nil, err
	}
	return &contact, nil
}

func (c Contact) MarshalText() (text []byte, err error) {
	var parts []string
	if c.Name != "" {
		parts = append(parts, c.Name)
	}
	if c.SecondName != "" {
		parts = append(parts, c.SecondName)
	}
	if c.LastName != "" {
		parts = append(parts, c.LastName)
	}
	return []byte(strings.Join(parts, " ")), nil
}
