package yameta

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix/client"
	"github.com/mitchellh/mapstructure"
	"reflect"
	"time"
)

type Deal struct {
	ID                       string           `bitrix:"ID"`
	Title                    string           `bitrix:"TITLE"`
	DealStage                *DirectoryEntree `bitrix:"STAGE_ID" yson:"Стадия сделки"`
	RoomID                   string           `bitrix:"UF_CRM_ROOM" yson:"Номер"`
	PlannedCheckin           *time.Time       `bitrix:"UF_CRM_CHECK_IN" yson:"Заезд"`
	ActualCheckin            *time.Time       `bitrix:"UF_CRM_ACTUAL_CHECK_IN" yson:"Регистрация заезда"`
	PlannedCheckout          *time.Time       `bitrix:"UF_CRM_CHECK_OUT" yson:"Выезд"`
	ActualCheckout           *time.Time       `bitrix:"UF_CRM_ACTUAL_CHECK_OUT" yson:"Регистрация выезда"`
	ActivationID             string           `bitrix:"UF_CRM_1624293706841" yson:"-"`
	FoodPromo                string           `bitrix:"UF_CRM_1622032303674" yson:"-"`
	TaxiPromo                string           `bitrix:"UF_CRM_1622032314072" yson:"-"`
	LavkaPromo               string           `bitrix:"UF_CRM_1622032332118" yson:"-"`
	PlusPromo                string           `bitrix:"UF_CRM_1622032352962" yson:"-"`
	HotelPromo               string           `bitrix:"UF_CRM_1623146087462" yson:"-"`
	ReturnCustomer           bool             `bitrix:"IS_RETURN_CUSTOMER" yson:"Повторная сделка"`
	Closed                   bool             `bitrix:"CLOSED" yson:"Сделка закрыта"`
	Sum                      float64          `bitrix:"OPPORTUNITY" yson:"Сумма"`
	Currency                 string           `bitrix:"CURRENCY_ID" yson:"Валюта"`
	Source                   *DirectoryEntree `bitrix:"SOURCE_ID" yson:"Источник"`
	PrepaySum                *float64         `bitrix:"UF_CRM_PREPAY_SUM" yson:"Сумма к предоплате"`
	PaidSum                  *float64         `bitrix:"UF_CRM_PAID_SUM" yson:"Фактически оплачено"`
	PaymentSystemName        string           `bitrix:"UF_CRM_PAYMENT_SYSTEM_NAME" yson:"Название способа оплаты"`
	NumAdults                *int             `bitrix:"UF_CRM_ADULTS" yson:"Кол-во взрослых"`
	NumChildren              *int             `bitrix:"UF_CRM_CHILDREN" yson:"Кол-во детей"`
	Days                     *int             `bitrix:"UF_CRM_DAYS" yson:"Кол-во дней проживания"`
	Nights                   *int             `bitrix:"UF_CRM_NIGHTS" yson:"Кол-во ночей проживания"`
	RoomTypeName             string           `bitrix:"UF_CRM_ROOM_TYPE_NAME" yson:"Название категории номера"`
	PlacementName            string           `bitrix:"UF_CRM_PLACEMENT_NAME" yson:"Способ размещения"`
	RatePlanName             string           `bitrix:"UF_CRM_RATE_PLAN_NAME" yson:"Название тарифа"`
	CustomerComment          string           `bitrix:"UF_CRM_CUSTOMER_COMMENT" yson:"Комментарий гостя"`
	TrafficSource            string           `bitrix:"UF_CRM_UTM_SOURCE" yson:"Источник трафика"`
	TrafficType              string           `bitrix:"UF_CRM_UTM_MEDIUM" yson:"Тип трафика"`
	MarketingCampaign        string           `bitrix:"UF_CRM_UTM_CAMPAIGN" yson:"Маркетинговая кампания"`
	DistributionSubscription bool             `bitrix:"UF_CRM_MAIL_DISTRIBUTION_SUBSCRIPTION" yson:"Согласие на получение рассылки"`
	ChannelName              string           `bitrix:"UF_CRM_CHANNEL_NAME" yson:"OTA источник бронирования"`
	BookingNumber            string           `bitrix:"UF_CRM_BOOKING_NUMBER" yson:"№ бронирования (без идентификатора номера)"`
	CreatedAt                *time.Time       `bitrix:"DATE_CREATE" yson:"Дата создания"`
	ContactID                string           `bitrix:"CONTACT_ID" yson:"-"`
	Contact                  *Contact         `bitrix:"-" yson:"Контакт"`
}

func (d *Deal) GetCheckin() *time.Time {
	if d.ActualCheckin != nil {
		return d.ActualCheckin
	}
	return d.PlannedCheckin
}

func (d *Deal) GetCheckout() *time.Time {
	if d.ActualCheckout != nil {
		return d.ActualCheckout
	}
	return d.PlannedCheckout
}

type dealFieldsStruct struct {
	RoomID       dealField
	DealStage    dealField
	ActivationID dealField
}

type dealFieldName string

type dealField struct {
	Name       dealFieldName
	BitrixName client.DealField
}

var allFieldsMap map[dealFieldName]client.DealField
var allFieldsSlice []client.DealField
var dealFields dealFieldsStruct

func init() {
	dt := reflect.TypeOf(Deal{})
	allFieldsMap = make(map[dealFieldName]client.DealField, dt.NumField())
	for i := 0; i < dt.NumField(); i++ {
		fieldName := dt.Field(i).Name
		attrName := dt.Field(i).Tag.Get("bitrix")
		if attrName == "" {
			attrName = fieldName
		}
		fn := client.DealField(attrName)
		allFieldsMap[dealFieldName(fieldName)] = fn
		allFieldsSlice = append(allFieldsSlice, fn)
	}
	dealFields = dealFieldsStruct{}
	ft := reflect.TypeOf(dealFields)
	fv := reflect.ValueOf(&dealFields).Elem()
	for i := 0; i < ft.NumField(); i++ {
		fieldName := dealFieldName(ft.Field(i).Name)
		if btrField, exists := allFieldsMap[fieldName]; exists {
			df := dealField{}
			dfv := reflect.ValueOf(&df).Elem()
			dfv.FieldByName("Name").SetString(string(fieldName))
			dfv.FieldByName("BitrixName").SetString(string(btrField))
			fv.Field(i).Set(dfv)
		} else {
			panic("unknown filter field " + fieldName)
		}
	}
}

func (c *Client) mapToDeal(sourceMap map[string]interface{}) (*Deal, error) {
	if err := c.ensureDirectory(); err != nil {
		return nil, err
	}
	var deal Deal
	decoder, err := mapstructure.NewDecoder(&mapstructure.DecoderConfig{
		Metadata: nil,
		DecodeHook: mapstructure.ComposeDecodeHookFunc(
			mapstructure.StringToTimeHookFunc(time.RFC3339),
			stringToBoolHookFunc(),
			stringToFloat64HookFunc(),
			stringToIntHookFunc(),
			stringToDirectoryHookFunc(c.directory),
			emptyStringToNilPointerHookFunc()),
		TagName: "bitrix",
		Result:  &deal,
	})
	if err != nil {
		return nil, err
	}
	if err := decoder.Decode(sourceMap); err != nil {
		return nil, err
	}
	return &deal, nil
}
