package yameta

import (
	"fmt"
	"github.com/mitchellh/mapstructure"
	"math"
	"reflect"
	"strconv"
	"strings"
)

func stringToBoolHookFunc() mapstructure.DecodeHookFunc {
	return func(
		f reflect.Type,
		t reflect.Type,
		data interface{}) (interface{}, error) {
		if f.Kind() != reflect.String {
			return data, nil
		}
		if t.Kind() != reflect.Bool {
			return data, nil
		}

		switch data.(string) {
		case "Y", "y", "1":
			return true, nil
		case "N", "n", "0":
			return false, nil
		default:
			return "", fmt.Errorf("failed to convert '%s' to boolean", data.(string))
		}
	}
}

func emptyStringToNilPointerHookFunc() mapstructure.DecodeHookFunc {
	return func(
		f reflect.Type,
		t reflect.Type,
		data interface{}) (interface{}, error) {
		if f.Kind() == reflect.String && t.Kind() == reflect.Ptr && data.(string) == "" {
			return nil, nil
		} else {
			return data, nil
		}
	}
}

func stringToIntHookFunc() mapstructure.DecodeHookFunc {
	return func(
		f reflect.Type,
		t reflect.Type,
		data interface{}) (interface{}, error) {
		if f.Kind() != reflect.String {
			return data, nil
		}
		if t.Kind() != reflect.Int {
			return data, nil
		}

		strVar := data.(string)
		value, err := strconv.ParseInt(strVar, 10, 64)
		if err != nil {
			return 0.0, fmt.Errorf("failed to convert '%s' to int: %w", strVar, err)
		}
		return int(value), nil
	}
}

func stringToFloat64HookFunc() mapstructure.DecodeHookFunc {
	return func(
		f reflect.Type,
		t reflect.Type,
		data interface{}) (interface{}, error) {
		if f.Kind() != reflect.String {
			return data, nil
		}
		if t.Kind() != reflect.Float64 {
			return data, nil
		}

		strVar := data.(string)
		strVar = strings.TrimSuffix(strVar, "|RUB")
		value, err := strconv.ParseFloat(strVar, 32)
		if err != nil {
			return 0.0, fmt.Errorf("failed to convert '%s' to float32: %w", strVar, err)
		}
		value = Round(value, 0.01)
		return value, nil
	}
}

func stringToDirectoryHookFunc(dir map[string]DirectoryEntree) mapstructure.DecodeHookFunc {
	return func(
		f reflect.Type,
		t reflect.Type,
		data interface{}) (interface{}, error) {
		if f.Kind() != reflect.String {
			return data, nil
		}
		if t == reflect.TypeOf(DirectoryEntree{}) {
			strVar := data.(string)
			entry, exists := dir[strVar]
			if !exists {
				return DirectoryEntree{}, fmt.Errorf("directory entry id %s is not found", strVar)
			}
			return entry, nil
		}
		return data, nil
	}
}

func Round(x, unit float64) float64 {
	return math.Round(x/unit) * unit
}
