package alice4business

import (
	"bytes"
	"encoding/json"
	"fmt"
	"io/ioutil"
	"net/http"
	"net/url"
	"time"
)

type Client struct {
	config Config
}

const ok = "ok"
const aliceTimeFormat = "2006-01-02 15:04:05"

func (c *Client) GetRoom(externalID string) (*RoomInfo, error) {
	var result roomInfoResponse
	if err := c.get("/api/public/rooms/info", url.Values{"external_room_id": {externalID}}, &result); err != nil {
		return nil, err
	}
	if result.Status != ok {
		return nil, fmt.Errorf("unexpected status in status response: %s", result.Status)
	}
	return &result.Info, nil
}

func (c *Client) GetDevice(externalID string) (*DeviceInfo, error) {
	var result deviceInfoResponse
	if err := c.get("/api/public/devices/info", url.Values{"external_id": {externalID}}, &result); err != nil {
		return nil, err
	}
	if result.Status != ok {
		return nil, fmt.Errorf("unexpected status in status response: %s", result.Status)
	}
	return &result.Info, nil
}

func (c *Client) ResetRoom(externalID string) (string, error) {
	var result startOperationResponse
	if err := c.post("/api/public/rooms/reset", map[string]string{"external_room_id": externalID}, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", fmt.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.OperationID, nil
}

func (c *Client) ActivateRoom(externalID string, applyPromoCode bool) (string, error) {
	var result startOperationResponse
	if err := c.post("/api/public/rooms/activate",
		map[string]interface{}{
			"external_room_id": externalID,
			"with_promocode":   applyPromoCode,
		}, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", fmt.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.OperationID, nil
}

func (c *Client) ActivateDevice(externalID string, applyPromoCode bool) (string, error) {
	var result startOperationResponse
	if err := c.post("/api/public/devices/activate",
		map[string]interface{}{
			"external_id":    externalID,
			"with_promocode": applyPromoCode,
		}, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", fmt.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.OperationID, nil
}

func (c *Client) ResetDevice(externalID string) (string, error) {
	var result startOperationResponse
	if err := c.post("/api/public/devices/reset", map[string]string{"external_id": externalID}, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", fmt.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.OperationID, nil
}

func (c *Client) CreateRoomActivation(externalID string, firstActivation *time.Time, lastActivation *time.Time) (string, error) {
	if firstActivation == nil || lastActivation == nil {
		return "", fmt.Errorf("activation time boundaries must be set")
	}
	payload := map[string]string{
		"external_id":           externalID,
		"first_activation_date": firstActivation.Format(aliceTimeFormat),
		"last_activation_date":  lastActivation.Format(aliceTimeFormat),
	}
	var result activationResponse
	if err := c.post("/api/public/rooms/activation", payload, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", fmt.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.ID, nil
}

func (c *Client) CreateDeviceActivation(externalID string, checkin time.Time, checkout time.Time) (string, error) {
	payload := map[string]string{
		"external_id":           externalID,
		"first_activation_date": checkin.Format(aliceTimeFormat),
		"last_activation_date":  checkout.Format(aliceTimeFormat),
	}
	var result activationResponse
	if err := c.post("/api/public/devices/activation", payload, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", fmt.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.ID, nil
}

func (c *Client) get(method string, args url.Values, result interface{}) error {
	request, err := http.NewRequest("GET", fmt.Sprintf("%s%s", c.config.APIEndpoint, method), nil)
	if err != nil {
		return fmt.Errorf("error while building http request: %w", err)
	}

	request.URL.RawQuery = args.Encode()
	request.Header = http.Header{}
	request.Header["Authorization"] = []string{fmt.Sprintf("OAuth %s", c.config.OAuthToken)}
	resp, err := http.DefaultClient.Do(request)
	if err != nil {
		return fmt.Errorf("unable to execute http request: %w", err)
	}
	defer func() { _ = resp.Body.Close() }()
	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return fmt.Errorf("unable to read response body: %w", err)
	}
	if resp.StatusCode != 200 {
		return fmt.Errorf("unexpected HTTP status %d, response is %+v", resp.StatusCode, string(data))
	}
	err = json.Unmarshal(data, result)
	return err
}

func (c *Client) post(method string, body interface{}, result interface{}) error {
	jsonValue, err := json.Marshal(body)
	if err != nil {
		return err
	}
	request, err := http.NewRequest("POST", fmt.Sprintf("%s%s", c.config.APIEndpoint, method),
		bytes.NewBuffer(jsonValue))
	if err != nil {
		return fmt.Errorf("error while building http request: %w", err)
	}
	request.Header = http.Header{}
	request.Header.Set("Content-Type", "application/json")
	request.Header["Authorization"] = []string{fmt.Sprintf("OAuth %s", c.config.OAuthToken)}
	resp, err := http.DefaultClient.Do(request)
	if err != nil {
		return fmt.Errorf("unable to execute http request: %w", err)
	}
	defer func() { _ = resp.Body.Close() }()
	data, err := ioutil.ReadAll(resp.Body)
	if err != nil {
		return fmt.Errorf("unable to read response body: %w", err)
	}
	if resp.StatusCode/100 != 2 {
		return fmt.Errorf("unexpected HTTP status %d, response is %+v", resp.StatusCode, string(data))
	}
	err = json.Unmarshal(data, result)
	return err
}

func NewClient(cfg Config) *Client {
	return &Client{config: cfg}
}
