package client

import (
	"context"
	"fmt"
	"github.com/go-resty/resty/v2"
	"net/http"
)

type objectList struct {
	Result []map[string]interface{} `json:"result"`
	Total  int                      `json:"total"`
	Next   *int                     `json:"next,omitempty"`
}

type DealField string
type DealStage string

type Client interface {
	List(ctx context.Context, query *listQuery) (*objectList, error)
	ListAll(ctx context.Context, query *listQuery, mapper Mapper) ([]interface{}, error)
	GetPager(query *listQuery, mapper Mapper) *Pager
	Get(ctx context.Context, objectType string, id string) (map[string]interface{}, error)
	Update(ctx context.Context, query *updateQuery) error
}

type client struct {
	cfg        Config
	httpClient *resty.Client
}

type updateResult struct {
	Result bool `json:"result"`
}

type objectResult struct {
	Result map[string]interface{} `json:"result"`
}

func (c client) GetPager(query *listQuery, mapper Mapper) *Pager {
	p := Pager{
		client: c,
		query:  *query,
		mapper: mapper,
	}
	return &p
}

func (c client) List(ctx context.Context, query *listQuery) (*objectList, error) {
	var result objectList
	if err := c.httpPost(ctx, fmt.Sprintf("%s.list", query.objectType), query.toMap(), &result); err != nil {
		return nil, err
	}
	return &result, nil
}

func (c client) Update(ctx context.Context, query *updateQuery) error {
	var result updateResult
	if err := c.httpPost(ctx, fmt.Sprintf("%s.update", query.objectType), query.toMap(), &result); err != nil {
		return err
	}
	if !result.Result {
		return fmt.Errorf("unexpected non-true result of update call")
	}
	return nil
}

func (c client) Get(ctx context.Context, objectType string, id string) (map[string]interface{}, error) {
	var result objectResult
	query := map[string]string{
		"id": id,
	}
	if err := c.httpPost(ctx, fmt.Sprintf("%s.get", objectType), query, &result); err != nil {
		return nil, err
	}
	return result.Result, nil
}

func (c client) ListAll(ctx context.Context, query *listQuery, mapper Mapper) ([]interface{}, error) {
	var result []interface{}
	pager := c.GetPager(query, mapper)
	for {
		items, hasNext, err := pager.NextPage(ctx)
		if err != nil {
			return nil, err
		}
		result = append(result, items...)
		if !hasNext {
			break
		}
	}
	return result, nil
}

func NewHTTPClient(cfg Config) Client {
	return client{
		cfg:        cfg,
		httpClient: resty.New().SetTimeout(cfg.Timeout),
	}
}

func (c *client) httpPost(ctx context.Context, method string, body interface{}, result interface{}) error {
	if c.cfg.Secret == "" {
		return fmt.Errorf("bitrix secret is not specified")
	}
	url := fmt.Sprintf("%s/%s/%s.json", c.cfg.RootURL, c.cfg.Secret, method)
	resp, err := c.httpClient.R().
		SetBody(body).
		SetResult(result).
		Post(url)
	if err != nil {
		return fmt.Errorf("error while posting form: %w", err)
	}

	if resp.StatusCode() != 200 {
		return UnexpectedHTTPStatusError{
			StatusCode:   resp.StatusCode(),
			ResponseBody: resp.RawResponse,
		}
	}

	return err
}

type UnexpectedHTTPStatusError struct {
	StatusCode   int
	ResponseBody *http.Response
}

func (e UnexpectedHTTPStatusError) Error() string {
	return fmt.Sprintf("unexpected HTTP status %d, response is %+v", e.StatusCode, e.ResponseBody)
}
