package client

import "fmt"

type listQuery struct {
	objectType  string
	filters     []filter
	fields      []DealField
	allStandard bool
	allUser     bool
	start       int
}

type filterOperation int

const (
	eq filterOperation = iota
	gt
	lt
	ne
	in
)

func (f filterOperation) String() string {
	return [...]string{"", ">", "<", "!", "@"}[f]
}

type filter struct {
	op    filterOperation
	field DealField
	value interface{}
}

func (q *listQuery) EQ(field DealField, value interface{}) *listQuery {
	q.filters = append(q.filters, filter{
		op:    eq,
		field: field,
		value: value,
	})
	return q
}

func (q *listQuery) NE(field DealField, value interface{}) *listQuery {
	q.filters = append(q.filters, filter{
		op:    ne,
		field: field,
		value: value,
	})
	return q
}

func (q *listQuery) GT(field DealField, value interface{}) *listQuery {
	q.filters = append(q.filters, filter{
		op:    gt,
		field: field,
		value: value,
	})
	return q
}

func (q *listQuery) LT(field DealField, value interface{}) *listQuery {
	q.filters = append(q.filters, filter{
		op:    lt,
		field: field,
		value: value,
	})
	return q
}

func (q *listQuery) Empty(field DealField) *listQuery {
	q.filters = append(q.filters, filter{
		op:    eq,
		field: field,
		value: nil,
	})
	return q
}

func (q *listQuery) NotEmpty(field DealField) *listQuery {
	q.filters = append(q.filters, filter{
		op:    ne,
		field: field,
		value: nil,
	})
	return q
}

func (q *listQuery) IN(field DealField, values ...interface{}) *listQuery {
	q.filters = append(q.filters, filter{
		op:    in,
		field: field,
		value: values,
	})
	return q
}

func (q *listQuery) Select(field DealField) *listQuery {
	q.fields = append(q.fields, field)
	return q
}

func (q *listQuery) SelectFields(fields ...DealField) *listQuery {
	q.fields = append(q.fields, fields...)
	return q
}

func (q *listQuery) SelectAll() *listQuery {
	q.allStandard = true
	q.allUser = true
	return q
}

func (q *listQuery) SelectAllStandard() *listQuery {
	q.allStandard = true
	return q
}

func (q *listQuery) SelectAllUser() *listQuery {
	q.allUser = true
	return q
}

func (q *listQuery) ClearSelect() *listQuery {
	q.allUser = false
	q.allStandard = false
	q.fields = q.fields[:0]
	return q
}

func (q *listQuery) toMap() map[string]interface{} {
	var selects []string
	filters := map[string]interface{}{}
	if q.allStandard {
		selects = append(selects, "*")
	}
	if q.allUser {
		selects = append(selects, "UF_*")
	}
	for _, f := range q.fields {
		selects = append(selects, string(f))
	}
	for _, f := range q.filters {
		key := fmt.Sprintf("%s%s", f.op.String(), f.field)
		filters[key] = f.value
	}
	return map[string]interface{}{
		"select": selects,
		"filter": filters,
		"start":  fmt.Sprintf("%d", q.start),
	}
}

func NewListQuery(objectType string) *listQuery {
	return &listQuery{
		objectType: objectType,
	}
}
