package bitrix

import (
	"a.yandex-team.ru/travel/budapest/bitrix_sync/pkg/bitrix/client"
	"fmt"
	"github.com/mitchellh/mapstructure"
	"reflect"
	"strings"
	"time"
)

type HotelSchema struct {
	modelType         reflect.Type
	remappedModelType reflect.Type
	AllFields         []client.DealField
}

func LoadWellKnownFields(cfg Config, wellKnownFields interface{}) error {
	var df client.DealField
	dft := reflect.TypeOf(df)
	dealFieldsType := reflect.TypeOf(wellKnownFields).Elem()
	dealFieldsValue := reflect.ValueOf(wellKnownFields).Elem()
	for i := 0; i < dealFieldsType.NumField(); i++ {
		fieldVal := dealFieldsValue.Field(i)
		field := dealFieldsType.Field(i)
		if field.Type != dft {
			return InvalidWellKnownField{
				FieldName: field.Name,
				FieldType: field.Type,
			}
		}
		if bitrixField, exists := cfg.DealFields[field.Name]; exists {
			fieldVal.SetString(bitrixField)
		} else {
			return NonConfiguredWellKnownField{FieldName: field.Name}
		}
	}
	return nil
}

func LoadWellKnownStages(cfg Config, wellKnownStages interface{}) error {
	var ds client.DealStage
	dst := reflect.TypeOf(ds)
	dealStagesType := reflect.TypeOf(wellKnownStages).Elem()
	dealStagesValue := reflect.ValueOf(wellKnownStages).Elem()
	for i := 0; i < dealStagesType.NumField(); i++ {
		fieldVal := dealStagesValue.Field(i)
		field := dealStagesType.Field(i)
		if field.Type != dst {
			return InvalidWellKnownStage{
				FieldName: field.Name,
				FieldType: field.Type,
			}
		}
		if btrStage, exists := cfg.DealStages[field.Name]; exists {
			fieldVal.SetString(btrStage)
		} else {
			return NonConfiguredWellKnownStage{StageName: field.Name}
		}
	}
	return nil
}

func LoadMappingSchema(cfg Config, modelType reflect.Type) (*HotelSchema, error) {
	schema := HotelSchema{modelType: modelType}
	allFieldsMap := make(map[string]client.DealField, modelType.NumField())
	var allFieldsSlice []client.DealField
	for i := 0; i < modelType.NumField(); i++ {
		fieldName := modelType.Field(i).Name
		bitrixName := modelType.Field(i).Tag.Get("bitrix")
		if bitrixName == "" || bitrixName == "-" {
			continue
		}
		if bitrixName == "!" {
			var exists bool
			bitrixName, exists = cfg.DealFields[fieldName]
			if !exists {
				return nil, fmt.Errorf("bitrix field for deal attribute %s is not specified", fieldName)
			}
		}
		bitrixField := client.DealField(bitrixName)
		allFieldsMap[fieldName] = bitrixField
		allFieldsSlice = append(allFieldsSlice, bitrixField)
	}

	newFields := make([]reflect.StructField, 0)
	for i := 0; i < modelType.NumField(); i++ {
		attrName := modelType.Field(i).Tag.Get("bitrix")
		newFields = append(newFields, modelType.Field(i))
		if attrName == "!" {
			bitrixFieldName := allFieldsMap[modelType.Field(i).Name]
			newTag := strings.Replace(string(modelType.Field(i).Tag), "!", string(bitrixFieldName), 1)
			newFields[i].Tag = reflect.StructTag(newTag)
		}
	}
	schema.remappedModelType = reflect.StructOf(newFields)
	schema.AllFields = allFieldsSlice
	return &schema, nil
}

func (c *HotelSchema) MapToDeal(sourceMap map[string]interface{}, directory Directory) (interface{}, error) {
	mappedDeal := reflect.New(c.remappedModelType)
	decodeHookFunctions := []mapstructure.DecodeHookFunc{
		mapstructure.StringToTimeHookFunc(time.RFC3339),
		stringToBoolHookFunc(),
		stringToFloat64HookFunc(),
		stringToIntHookFunc(),
	}
	if directory != nil {
		decodeHookFunctions = append(decodeHookFunctions, stringToDirectoryHookFunc(directory))
	}
	decodeHookFunctions = append(decodeHookFunctions, emptyStringToNilPointerHookFunc())
	decoder, err := mapstructure.NewDecoder(&mapstructure.DecoderConfig{
		Metadata:   nil,
		DecodeHook: mapstructure.ComposeDecodeHookFunc(decodeHookFunctions...),
		TagName:    "bitrix",
		Result:     mappedDeal.Interface(),
	})
	if err != nil {
		return nil, err
	}
	if err := decoder.Decode(sourceMap); err != nil {
		return nil, err
	}
	d := mappedDeal.Elem().Convert(c.modelType).Interface()
	return d, nil
}
