package tgalerts

import (
	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/travel/hotels/lib/go/ytstorage"
	"context"
	tb "gopkg.in/tucnak/telebot.v2"
	"reflect"
	"time"
)

type Alerter struct {
	bot              *tb.Bot
	admins           map[string]bool
	channelsToNotify map[int64]*alertChannel
	tgConfig         Config
	ytConfig         ytstorage.Config
	logger           log.Logger
	suspendedAlerts  map[interface{}]time.Time
}

type alertChannel struct {
	ID   int64
	Name string
}

func NewAlerter(ctx context.Context, tgConfig Config, ytConfig ytstorage.Config, logger log.Logger) (*Alerter, error) {
	botSettings := tb.Settings{
		Token:  tgConfig.TGToken,
		Poller: &tb.LongPoller{Timeout: 5 * time.Second},
	}
	bot, err := tb.NewBot(botSettings)
	if err != nil {
		return nil, err
	}
	channels := map[int64]*alertChannel{}
	if tgConfig.YTPath != "" {
		chanList, err := ytstorage.Load(ctx, reflect.TypeOf(alertChannel{}), tgConfig.YTPath, ytConfig.Token, ytConfig.Proxy)
		if err != nil {
			logger.Error("Error while loading tg alert channels", log.Error(err))
			channels = make(map[int64]*alertChannel)
		} else {
			for _, ch := range chanList {
				channels[ch.(*alertChannel).ID] = ch.(*alertChannel)
			}
		}
	}
	admins := map[string]bool{}
	for _, admin := range tgConfig.AdminLogins {
		admins[admin] = true
	}

	alerter := Alerter{
		bot:              bot,
		admins:           admins,
		channelsToNotify: channels,
		tgConfig:         tgConfig,
		ytConfig:         ytConfig,
		logger:           logger,
		suspendedAlerts:  make(map[interface{}]time.Time),
	}
	bot.Handle("/startalerts", alerter.handleStartAlerts)
	bot.Handle("/stopalerts", alerter.handleStopAlerts)
	go func() {
		logger.Info("Starting tg bot")
		bot.Start()
		logger.Info("TG bot stopped")
	}()
	return &alerter, nil
}

func (a *Alerter) saveChannels() error {
	var channels []*alertChannel
	for _, ch := range a.channelsToNotify {
		channels = append(channels, ch)
	}
	return ytstorage.Save(context.Background(), channels, a.tgConfig.YTPath, 1, a.ytConfig.Token, a.ytConfig.Proxy)
}

func (a *Alerter) handleStartAlerts(message *tb.Message) {
	if _, allowed := a.admins[message.Sender.Username]; allowed {
		if _, alreadyStarted := a.channelsToNotify[message.Chat.ID]; alreadyStarted {
			_, _ = a.bot.Send(message.Chat, "Уже включено")
			return
		} else {
			a.channelsToNotify[message.Chat.ID] = &alertChannel{
				ID:   message.Chat.ID,
				Name: message.Chat.Title,
			}
			err := a.saveChannels()
			if err != nil {
				_, _ = a.bot.Send(message.Chat, "Произошла ошибка :(")
				a.logger.Error("Error while starting alerts", log.Error(err))
			} else {
				_, _ = a.bot.Send(message.Chat, "Включено!")
			}
		}
	} else {
		_, _ = a.bot.Send(message.Chat, "У вас не достаточно прав")
	}
}

func (a *Alerter) handleStopAlerts(message *tb.Message) {
	if _, allowed := a.admins[message.Sender.Username]; allowed {
		if _, exists := a.channelsToNotify[message.Chat.ID]; exists {
			delete(a.channelsToNotify, message.Chat.ID)
			err := a.saveChannels()
			if err != nil {
				_, _ = a.bot.Send(message.Chat, "Произошла ошибка :(")
				a.logger.Error("Error while stopping alerts", log.Error(err))
			} else {
				_, _ = a.bot.Send(message.Chat, "Выключено!")
			}
		} else {
			_, _ = a.bot.Send(message.Chat, "Уже выключено")
		}
	} else {
		_, _ = a.bot.Send(message.Chat, "У вас не достаточно прав")
	}
}

func (a *Alerter) Stop() {
	a.bot.Stop()
}

func (a *Alerter) Alert(text string) {
	for id := range a.channelsToNotify {
		if _, err := a.bot.Send(tb.ChatID(id), text); err != nil {
			a.logger.Error("Error while sending message alert", log.Error(err), log.String("text", text))
		}
	}
}

func (a *Alerter) NonRepeatableAlert(text string, key interface{}, timeout time.Duration) {
	deadline, exists := a.suspendedAlerts[key]
	if exists {
		if deadline.After(time.Now()) {
			a.logger.Info("Alert was already shown", log.Any("key", key))
			return
		}
	}
	for id := range a.channelsToNotify {
		if _, err := a.bot.Send(tb.ChatID(id), text); err != nil {
			a.logger.Error("Error while sending message alert", log.Error(err), log.String("text", text))
		}
	}
	a.suspendedAlerts[key] = time.Now().Add(timeout)
}
