package main

import (
	"context"
	"time"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/app"
	"a.yandex-team.ru/travel/budapest/metapms/internal/events"
	"a.yandex-team.ru/travel/budapest/metapms/internal/queue"
	"a.yandex-team.ru/travel/library/go/configuration"
	sandboxplanner "a.yandex-team.ru/travel/library/go/sandbox_planner"
)

type Config struct {
	Env              string `config:"env,required,backend=flags"`
	DatabaseUser     string `config:"db_user,required"`
	DatabasePassword string `config:"db_password,required"`
	DatabaseName     string `config:"db_name,required"`
	DatabasePort     int    `config:"db_port,required"`
	HotelKey         string `config:"hotel,backend=flags"`
}

// TODO(tivelkov): dynamically fetch these from YC using https://nda.ya.ru/t/U6IfLlMQ4tAzKJ
var hostsMap = map[string][]string{
	"testing": {"sas-5y32xff7mj4upj35.db.yandex.net", "vla-hm3wzg34w8qdwdrp.db.yandex.net"},
	"prod":    {"sas-edz3q9e16tfw6cya.db.yandex.net", "vla-dlrl0p73cetvn2mc.db.yandex.net"},
	"local":   {"localhost"},
}

func main() {
	cfg := Config{
		DatabasePort: 6432,
		DatabaseName: "metapms",
		DatabaseUser: "metapms",
	}
	if err := sandboxplanner.ReplaceArgsFromEnv(); err != nil {
		panic(xerrors.Errorf("unable to initialize config environment: %w", err))
	}
	configLoader := configuration.NewDefaultConfitaLoader()
	if err := configLoader.Load(context.Background(), &cfg); err != nil {
		panic(xerrors.Errorf("unable to load config: %w", err))
	}
	hosts, found := hostsMap[cfg.Env]
	if !found {
		panic("Environment database is not known")
	}
	dbConfig := app.DBConfig{
		Hosts:              hosts,
		Port:               cfg.DatabasePort,
		User:               cfg.DatabaseUser,
		Password:           cfg.DatabasePassword,
		Name:               cfg.DatabaseName,
		HostsUpdateTimeout: 5 * time.Second,
	}
	pg, err := app.BuildPGClient(dbConfig)
	if err != nil {
		panic(xerrors.Errorf("unable to get database client: %w", err))
	}
	tx, err := pg.GetPrimary()
	if err != nil {
		panic(xerrors.Errorf("unable to connect to primary DB: %w", err))
	}
	event := events.StatusReportRequested{}
	if cfg.HotelKey != "" {
		event.HotelKey = cfg.HotelKey
	} else {
		event.All = true
	}
	if err := queue.Push(tx, "Activator", &event); err != nil {
		panic(xerrors.Errorf("unable to push message: %w", err))
	}
}
