package main

import (
	"context"
	"fmt"
	"os"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/budapest/metapms/internal/app"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
	"a.yandex-team.ru/travel/budapest/metapms/internal/queue"
	"a.yandex-team.ru/travel/budapest/metapms/internal/synchronizer"
	"a.yandex-team.ru/travel/library/go/configuration"
	"a.yandex-team.ru/travel/library/go/metrics"
	metricserver "a.yandex-team.ru/travel/library/go/metrics/server"
)

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	config := configuration.NewDefaultConfitaLoader()
	err := config.Load(ctx, &Cfg)

	if err != nil {
		fmt.Println("can not load configuration:", err)
		ctxCancel()
		os.Exit(1)
	}

	logger, err := app.NewLogger(Cfg.App.Logging)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		ctxCancel()
		os.Exit(1)
	}

	rootRegistry := metrics.NewRegistryWithDeployTags()
	metrics.RunPerfMetricsUpdater(rootRegistry, Cfg.App.Metrics.PerfMetricsRefreshInterval)

	pg, err := app.BuildPGClient(Cfg.App.DB)
	if err != nil {
		logger.Fatal("Failed to create PG client", log.Error(err))
	}
	if Cfg.App.DB.MigrateOnStart {
		db, err := pg.GetPrimary()
		if err != nil {
			logger.Fatal("Unable to get tx for migration", log.Error(err))
		}
		toMigrate := []interface{}{model.Sync{},
			model.Hotel{},
			model.Room{},
			model.Booking{},
			model.Guest{},
			model.RoomStay{},
		}
		toMigrate = append(toMigrate, queue.GetModelsToMigrate()...)
		if err := db.AutoMigrate(toMigrate...); err != nil {
			logger.Fatal("Unable to run migrations", log.Error(err))
		}
	}

	if err := Cfg.Sync.LoadPMSKeys(); err != nil {
		logger.Fatal("Unable to load pms keys", log.Error(err))
	}
	s := synchronizer.New(Cfg.Sync, Cfg.PMS, pg, logger, rootRegistry)

	wg := sync.WaitGroup{}
	wg.Add(2)

	go func() {
		defer wg.Done()
		logger.Info("Starting synchronizer")
		err = s.Run(ctx)
		if err != nil {
			logger.Fatal("Unable to start synchronizer", log.Error(err))
		}
		logger.Info("Synchronizer stopped")
	}()

	go func() {
		defer wg.Done()
		logger.Info("Starting metrics server")
		err = metricserver.RunMetricsHTTPServer(ctx, Cfg.App.Metrics, logger, rootRegistry)
		if err != nil {
			logger.Fatal("Error while running metrics server", log.Error(err))
		}
		logger.Info("Metrics server is stopped")
	}()

	wg.Wait()
	logger.Info("App stopped")

}
