package activator

import (
	"context"
	"database/sql"
	"sync"
	"time"

	"google.golang.org/protobuf/proto"
	"gorm.io/gorm"
	"gorm.io/gorm/clause"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/alice4business"
	"a.yandex-team.ru/travel/budapest/metapms/internal/events"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
	"a.yandex-team.ru/travel/budapest/metapms/internal/pgclient"
	"a.yandex-team.ru/travel/budapest/metapms/internal/queue"
)

type Activator struct {
	pg      *pgclient.PGClient
	logger  log.Logger
	cfg     Config
	queue   *queue.Queue
	a4b     A4B
	metrics *activatorMetrics
}

func (a *Activator) onCheckIn(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.RoomStayCheckedIn)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	return a.createOperation(tx, payloadCast.RoomStayId, nil, payloadCast.HotelKey, model.RoomStayStatusCheckedIn, model.OperationTypeActivate)
}

func (a *Activator) onCheckOut(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.RoomStayCheckedOut)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	return a.createOperation(tx, payloadCast.RoomStayId, nil, payloadCast.HotelKey, model.RoomStayStatusCheckedOut, model.OperationTypeReset)
}

func (a *Activator) onRoomChanged(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.RoomStayChangedRoom)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	oldRoomID := uint(payloadCast.OldRoomId)
	newRoomID := uint(payloadCast.NewRoomId)
	if err := a.createOperation(tx, payloadCast.RoomStayId, &oldRoomID, payloadCast.HotelKey, model.RoomStayStatusCheckedIn, model.OperationTypeReset); err != nil {
		return xerrors.Errorf("room change: unable to start reset operation of the old room: %w", err)
	}
	if err := a.createOperation(tx, payloadCast.RoomStayId, &newRoomID, payloadCast.HotelKey, model.RoomStayStatusCheckedIn, model.OperationTypeActivate); err != nil {
		return xerrors.Errorf("room change: unable to start activate operation of the new room: %w", err)
	}
	return nil
}

func (a *Activator) onCheckInCancelled(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.RoomStayCheckinCancelled)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	return a.createOperation(tx, payloadCast.RoomStayId, nil, payloadCast.HotelKey, model.RoomStayStatusNew, model.OperationTypeReset)
}

func (a *Activator) Run(ctx context.Context) error {
	if !a.cfg.Enabled {
		return nil
	}
	wg := sync.WaitGroup{}
	wg.Add(2)
	go func() {
		defer wg.Done()
		err := a.queue.Listen(ctx, a.cfg.QueuePollInterval)
		if err != nil {
			a.logger.Error("error in event listening loop", log.Error(err))
		}
	}()
	go func() {
		defer wg.Done()
		err := a.runOperationsPeriodically(ctx)
		if err != nil {
			a.logger.Error("error in activator operations loop", log.Error(err))
		}
	}()
	wg.Wait()
	return nil
}

func (a *Activator) runOperationsPeriodically(ctx context.Context) error {
	ticker := time.NewTicker(a.cfg.OperationPollInterval)
	for {
		select {
		case <-ticker.C:
			err := a.runOperations()
			if err != nil {
				if !pgclient.IsLockError(err) {
					a.logger.Error("Error while running activator operations", log.Error(err))
				}
				continue
			}
		case <-ctx.Done():
			return nil
		}
	}
}

func (a *Activator) runOperations() error {
	db, err := a.pg.GetPrimary()
	if err != nil {
		return xerrors.Errorf("unable to get db connection to run operations: %w", err)
	}
	return db.Transaction(func(tx *gorm.DB) error {
		operations, err := a.getPendingOperations(tx)
		if err != nil {
			return xerrors.Errorf("unable to list pending operations: %w", err)
		}
		opCountByType := make(map[model.OperationType]float64, len(model.AllOperationTypes))
		for _, ot := range model.AllOperationTypes {
			opCountByType[ot] = 0
		}
		for _, op := range operations {
			opCountByType[op.Type]++
		}
		for k, v := range opCountByType {
			a.metrics.pendingOperations[k].Set(v)
		}

		if len(operations) == 0 {
			return nil
		}
		wg := sync.WaitGroup{}
		for i := range operations {
			op := &operations[i]
			wg.Add(1)
			go func() {
				defer wg.Done()
				opErr := tx.Transaction(func(opTX *gorm.DB) error {
					err := a.runOperation(opTX, op)
					if err != nil {
						return xerrors.Errorf("unable to run operation: %w", err)
					}
					opTX.Session(&gorm.Session{FullSaveAssociations: true}).Omit("RoomBinding").
						Save(op)
					return nil
				})
				if opErr != nil {
					a.logger.Error("Unable to run operation",
						log.UInt("OperationID", op.ID),
						log.String("OperationType", string(op.Type)),
						log.UInt("RoomStayID", op.RoomStayID),
						log.UInt("RoomID", op.RoomBinding.RoomID),
						log.Error(opErr))
					op.State = model.OperationStateRuntimeError
					op.Error = opErr.Error()
					tx.Session(&gorm.Session{FullSaveAssociations: true}).Omit("RoomBinding").
						Save(op)
				}
			}()
		}
		wg.Wait()
		return nil
	})

}

func (a *Activator) getPendingOperations(tx *gorm.DB) ([]model.Operation, error) {
	var operations []model.Operation
	subquery := tx.
		Model(model.Operation{}).Select("min(id)").
		Where("State in ?", []model.OperationState{model.OperationStateNew, model.OperationStateRunning, model.OperationStateCancelling}).
		Group("room_binding_id")
	if err := tx.
		Clauses(clause.Locking{Strength: "UPDATE"}).
		Preload("A4BBindings").
		Preload("RoomBinding").
		Preload("RoomBinding.Room").
		Model(model.Operation{}).
		Where("ID in (?)", subquery).
		Where("State in ?", []model.OperationState{model.OperationStateNew, model.OperationStateRunning, model.OperationStateCancelling}).
		Where("check_after <= now()").
		Find(&operations).Error; err != nil {
		return nil, xerrors.Errorf("unable to load running operations: %w", err)
	}
	if len(operations) > 0 {
		opStateMap := make(map[model.OperationState][]uint, len(operations))
		for _, op := range operations {
			opStateMap[op.State] = append(opStateMap[op.State], op.ID)
		}
		a.logger.Debug("Running activator operations",
			log.Int("NumPending", len(operations)),
			log.Any("OperationsByState", opStateMap))
	}
	return operations, nil
}

func (a *Activator) runOperation(tx *gorm.DB, op *model.Operation) error {
	hasPending, err := a.updateA4BOperationBindings(tx, op)
	if err != nil {
		return err
	}

	if hasPending {
		a.logger.Debug("Operation still awaits for a4b operations to complete",
			log.UInt("OperationID", op.ID),
			log.String("State", string(op.State)),
			log.UInt("RoomStayID", op.RoomStayID))
		op.CheckAfter = time.Now().Add(a.cfg.LongRetryDuration)
		return nil
	}

	a4BRoomID := op.RoomBinding.A4BRoomID
	ctx := tx.Statement.Context
	room, err := a.a4b.GetRoom(ctx, a4BRoomID)
	if err != nil {
		return xerrors.Errorf("unable to get status for room: %w", err)
	}
	if room.InProgress {
		a.logger.Warn("Room is in progress, while no pending a4b operation bindings exist",
			log.UInt("OperationID", op.ID),
			log.String("State", string(op.State)),
			log.UInt("RoomStayID", op.RoomStayID))
		op.CheckAfter = time.Now().Add(a.cfg.LongRetryDuration)
		return nil
	}

	for _, d := range room.Devices {
		if d.InProgress {
			a.logger.Warn("Room's device is in progress, while no pending a4b operation bindings exist",
				log.UInt("OperationID", op.ID),
				log.String("State", string(op.State)),
				log.UInt("RoomStayID", op.RoomStayID),
				log.String("DeviceID", d.DeviceID))
			op.CheckAfter = time.Now().Add(a.cfg.LongRetryDuration)
			return nil
		}
	}

	if op.State == model.OperationStateCancelling {
		op.State = model.OperationStateCancelled
		a.logger.Info("Done waiting for pending operations, cancelled",
			log.UInt("OperationID", op.ID),
			log.UInt("RoomStayID", op.RoomStayID))
		a.notifyOperationCancelled(tx, op)
		return nil
	}

	switch op.Type {
	case model.OperationTypeActivate:
		return a.runActivationOperation(tx, op, room)
	case model.OperationTypeReset:
		return a.runResetOperation(tx, op, room)
	default:
		op.State = model.OperationStateFailed
		op.Error = "Unexpected operation type"
		a.notifyOperationFailed(tx, op)
	}
	return nil
}

func (a *Activator) runResetOperation(tx *gorm.DB, op *model.Operation, room *alice4business.RoomInfo) error {
	a4BRoomID := op.RoomBinding.A4BRoomID
	nextStay := a.getUpcomingStayForRoom(tx, op.RoomBinding)
	var partial bool
	if nextStay != nil && nextStay.CheckInDateTime.Valid {
		partial = nextStay.CheckInDateTime.Time.Before(time.Now().Add(a.cfg.MinIntervalForPartial))
		if partial {
			a.logger.Info("Reset operation: next checkin soon, will do partial reset",
				log.UInt("OperationID", op.ID),
				log.String("A4BRoomID", a4BRoomID),
				log.UInt("RoomStayID", op.RoomStayID),
				log.UInt("NextStayID", nextStay.ID),
				log.Time("NextStayCheckInTime", nextStay.CheckInDateTime.Time))
			a.metrics.partialFallbacks.Inc()
		} else {
			a.logger.Info("Reset operation: plenty of time till next checkin, will do full reset",
				log.UInt("OperationID", op.ID),
				log.String("A4BRoomID", a4BRoomID),
				log.UInt("RoomStayID", op.RoomStayID),
				log.UInt("NextStayID", nextStay.ID),
				log.Time("NextStayCheckInTime", nextStay.CheckInDateTime.Time))
		}
	} else {
		a.logger.Info("Reset operation: no next checkin planned, will do full reset",
			log.UInt("OperationID", op.ID),
			log.String("A4BRoomID", a4BRoomID),
			log.UInt("RoomStayID", op.RoomStayID))
	}
	ctx := tx.Statement.Context
	switch room.Status {
	case alice4business.RoomInactive:
		a.logger.Info("Reset operation: room is inactive, that's success",
			log.UInt("OperationID", op.ID),
			log.String("A4BRoomID", a4BRoomID),
			log.UInt("RoomStayID", op.RoomStayID))
		op.State = model.OperationStateSuccess
		a.notifyOperationSuccess(tx, op)
		return nil
	case alice4business.RoomActive:
		a.logger.Info("Reset operation: room is active, will start the Reset operation",
			log.UInt("OperationID", op.ID),
			log.String("A4BRoomID", a4BRoomID),
			log.UInt("RoomStayID", op.RoomStayID))
		if err := a.startA4BRoomReset(ctx, op, a4BRoomID, partial); err != nil {
			return err
		}
		op.State = model.OperationStateRunning
		op.CheckAfter = time.Now().Add(a.cfg.LongRetryDuration)
		return nil
	case alice4business.RoomReset:
		a.logger.Info("Reset operation: operation failed, will retry",
			log.UInt("OperationID", op.ID),
			log.String("A4BRoomID", a4BRoomID),
			log.UInt("RoomStayID", op.RoomStayID))
		if err := a.startA4BRoomReset(ctx, op, a4BRoomID, partial); err != nil {
			return err
		}
		op.State = model.OperationStateRunning
		op.CheckAfter = time.Now().Add(a.cfg.LongRetryDuration)
		return nil
	case alice4business.RoomMixedStatus:
		if op.State != model.OperationStateNew && nextStay != nil && nextStay.CheckInDateTime.Valid && nextStay.CheckInDateTime.Time.Before(time.Now().Add(a.cfg.MinIntervalForPartial)) {
			a.logger.Warn("Reset operation: mixed room state when the next checkin is soon",
				log.UInt("OperationID", op.ID),
				log.String("A4BRoomID", a4BRoomID),
				log.UInt("RoomStayID", op.RoomStayID))
			op.State = model.OperationStateFailed
			op.Error = "Mixed room state when the next checkin is soon"
			a.notifyOperationFailed(tx, op)
		} else {
			a.logger.Info("Reset operation: mixed room state on reset operation, will start the Reset operation",
				log.UInt("OperationID", op.ID),
				log.String("A4BRoomID", a4BRoomID),
				log.UInt("RoomStayID", op.RoomStayID))
			if err := a.startA4BRoomReset(ctx, op, a4BRoomID, partial); err != nil {
				return err
			}
			op.State = model.OperationStateRunning
			op.CheckAfter = time.Now().Add(a.cfg.LongRetryDuration)
			return nil
		}
	default:
		op.State = model.OperationStateFailed
		op.Error = "Unexpected state on reset"
		a.notifyOperationFailed(tx, op)
	}
	return nil
}

func (a *Activator) runActivationOperation(tx *gorm.DB, op *model.Operation, room *alice4business.RoomInfo) error {
	a4BRoomID := op.RoomBinding.A4BRoomID
	ctx := tx.Statement.Context
	switch room.Status {
	case alice4business.RoomActive:
		if op.State == model.OperationStateNew {
			a.logger.Warn("Activation operation: room is already active when starting operation, that's weird",
				log.UInt("OperationID", op.ID),
				log.String("A4BRoomID", a4BRoomID),
				log.UInt("RoomStayID", op.RoomStayID))
		} else {
			a.logger.Info("Activation operation: room is activated successfully",
				log.UInt("OperationID", op.ID),
				log.String("A4BRoomID", a4BRoomID),
				log.UInt("RoomStayID", op.RoomStayID),
				log.String("PromoStatus", string(room.PromoStatus)))
		}
		if room.PromoStatus == alice4business.RoomPromoNotAvailable {
			var devicesForPromo []string
			for _, d := range room.Devices {
				if !d.HasPlus {
					if d.ExternalID == "" {
						a.logger.Warn("Activation operation: device of a non-promo-available room has no external-id for direct control",
							log.UInt("OperationID", op.ID),
							log.String("A4BRoomID", a4BRoomID),
							log.UInt("RoomStayID", op.RoomStayID),
							log.String("DeviceID", d.DeviceID))
						op.State = model.OperationStateFailed
						op.Error = "Device of a non-promo-available room has no external-id for direct control"
						a.notifyOperationFailed(tx, op)
						return nil
					}
					devicesForPromo = append(devicesForPromo, d.ExternalID)
				}
			}
			a.logger.Warn("Activation operation: promo is not applied on room activation, will try to apply one-by-one",
				log.UInt("OperationID", op.ID),
				log.String("A4BRoomID", a4BRoomID),
				log.UInt("RoomStayID", op.RoomStayID),
				log.Array("DeicesToApply", devicesForPromo),
			)
			for _, d := range devicesForPromo {
				if err := a.a4b.ApplyPromoForDevice(tx.Statement.Context, d); err != nil {
					a.logger.Error("Error while trying to apply promo code directly to device",
						log.UInt("OperationID", op.ID),
						log.String("A4BRoomID", a4BRoomID),
						log.UInt("RoomStayID", op.RoomStayID),
						log.String("DeviceExternalID", d),
						log.Error(err))
					op.State = model.OperationStateFailed
					op.Error = "Error while trying to apply promo code directly to device"
					a.notifyOperationFailed(tx, op)
					return nil
				} else {
					a.logger.Info("Activation operation: promo code applied directly to device",
						log.UInt("OperationID", op.ID),
						log.String("A4BRoomID", a4BRoomID),
						log.UInt("RoomStayID", op.RoomStayID),
						log.String("DeviceExternalID", d))
					a.metrics.promoActivationsOnDevice.Inc()
				}
			}
		}
		a.notifyOperationSuccess(tx, op)
		op.State = model.OperationStateSuccess
	case alice4business.RoomReset: // ошибка сброса => повторяем сброс с partial-флагом
		a.logger.Warn("Activation operation: room reset failure when starting activation operation, will restart reset",
			log.UInt("OperationID", op.ID),
			log.String("A4BRoomID", a4BRoomID),
			log.UInt("RoomStayID", op.RoomStayID))
		if err := a.startA4BRoomReset(ctx, op, a4BRoomID, true); err != nil {
			return err
		}
		a.metrics.partialFallbacks.Inc()
		op.State = model.OperationStateRunning
		op.CheckAfter = time.Now().Add(a.cfg.LongRetryDuration)

	case alice4business.RoomInactive:
		a.logger.Info("Activation operation: room is inactive",
			log.UInt("OperationID", op.ID),
			log.String("A4BRoomID", a4BRoomID),
			log.UInt("RoomStayID", op.RoomStayID))
		if err := a.startA4BRoomActivate(ctx, op, a4BRoomID, true); err != nil {
			return err
		}
		op.State = model.OperationStateRunning
		op.CheckAfter = time.Now().Add(a.cfg.FastRetryDuration)
	case alice4business.RoomMixedStatus:
		var devicesToReset []string
		var devicesToActivate []string
		for _, deviceInfo := range room.Devices {
			if deviceInfo.ExternalID == "" {
				a.logger.Warn("Device of a mixed-status room has no external-id for direct control",
					log.UInt("OperationID", op.ID),
					log.String("A4BRoomID", a4BRoomID),
					log.UInt("RoomStayID", op.RoomStayID),
					log.String("DeviceID", deviceInfo.DeviceID))
				op.State = model.OperationStateFailed
				op.Error = "Device of a mixed-status room has no external-id for direct control"
				a.notifyOperationFailed(tx, op)
			}
			switch deviceInfo.Status {
			case alice4business.DeviceInactive:
				devicesToActivate = append(devicesToActivate, deviceInfo.ExternalID)
			case alice4business.DeviceReset:
				devicesToReset = append(devicesToReset, deviceInfo.ExternalID)
			}
		}
		if len(devicesToReset)+len(devicesToActivate) > 0 {
			a.logger.Info("Activation operation: mixed room status",
				log.UInt("OperationID", op.ID),
				log.String("A4BRoomID", a4BRoomID),
				log.UInt("RoomStayID", op.RoomStayID),
				log.Array("DeviceToReset", devicesToReset),
				log.Array("DeviceToActivate", devicesToActivate))

			for _, deviceID := range devicesToActivate {
				if err := a.startA4BDeviceActivation(ctx, op, deviceID, true); err != nil {
					return err
				}
			}
			for _, deviceID := range devicesToReset {
				if err := a.startA4BDeviceReset(ctx, op, deviceID); err != nil {
					return err
				}
			}
			op.State = model.OperationStateRunning
			op.CheckAfter = time.Now().Add(a.cfg.FastRetryDuration)
		}
	default:
		op.State = model.OperationStateFailed
		op.Error = "Unexpected state on activation"
		a.notifyOperationFailed(tx, op)
	}
	return nil
}

func (a *Activator) startA4BDeviceReset(ctx context.Context, op *model.Operation, a4BDeviceID string) error {
	opID, err := a.a4b.ResetDevice(ctx, a4BDeviceID)
	if err != nil {
		return xerrors.Errorf("unable to start reset device operation: %w", err)
	}
	a.logger.Info("Started A4B ResetDevice operation",
		log.UInt("OperationID", op.ID),
		log.String("A4BDeviceID", a4BDeviceID),
		log.UInt("RoomStayID", op.RoomStayID),
		log.String("A4BOperationID", opID),
		log.String("A4BOperationType", string(model.A4BOperationTypeResetDevice)),
	)
	op.A4BBindings = append(op.A4BBindings, &model.OperationBinding{
		OperationID:    op.ID,
		Index:          len(op.A4BBindings),
		A4BOperationID: opID,
		A4BStatus:      alice4business.OperationStatusPending,
		Type:           model.A4BOperationTypeResetDevice,
		Started:        time.Now(),
		Options: map[string]interface{}{
			"external_id": a4BDeviceID,
		},
	})
	a.metrics.a4bOperationStarts[model.A4BOperationTypeResetDevice].Inc()
	return nil
}

func (a *Activator) startA4BDeviceActivation(ctx context.Context, op *model.Operation, a4BDeviceID string, withPromo bool) error {
	opID, err := a.a4b.ActivateDevice(ctx, a4BDeviceID, withPromo)
	if err != nil {
		return xerrors.Errorf("unable to start activate device operation: %w", err)
	}
	a.logger.Info("Started A4B ActivateDevice operation",
		log.UInt("OperationID", op.ID),
		log.String("A4BDeviceID", a4BDeviceID),
		log.UInt("RoomStayID", op.RoomStayID),
		log.String("A4BOperationID", opID),
		log.String("A4BOperationType", string(model.A4BOperationTypeActivateDevice)),
		log.Bool("ApplyPromoCode", withPromo),
	)
	op.A4BBindings = append(op.A4BBindings, &model.OperationBinding{
		OperationID:    op.ID,
		Index:          len(op.A4BBindings),
		A4BOperationID: opID,
		A4BStatus:      alice4business.OperationStatusPending,
		Type:           model.A4BOperationTypeActivateDevice,
		Started:        time.Now(),
		Options: map[string]interface{}{
			"external_id":      a4BDeviceID,
			"apply_promo_code": withPromo,
		},
	})
	a.metrics.a4bOperationStarts[model.A4BOperationTypeActivateDevice].Inc()
	if withPromo {
		a.metrics.promoActivationsOnDevice.Inc()
	}
	return nil
}

func (a *Activator) startA4BRoomReset(ctx context.Context, op *model.Operation, a4BRoomID string, partial bool) error {
	opID, err := a.a4b.ResetRoom(ctx, a4BRoomID, partial)
	if err != nil {
		return xerrors.Errorf("unable to start reset room operation: %w", err)
	}
	a.logger.Info("Started A4B ResetRoom operation",
		log.UInt("OperationID", op.ID),
		log.String("A4BRoomID", a4BRoomID),
		log.UInt("RoomStayID", op.RoomStayID),
		log.String("A4BOperationID", opID),
		log.String("A4BOperationType", string(model.A4BOperationTypeResetRoom)),
		log.Bool("Partial", partial),
	)
	op.A4BBindings = append(op.A4BBindings, &model.OperationBinding{
		OperationID:    op.ID,
		Index:          len(op.A4BBindings),
		A4BOperationID: opID,
		A4BStatus:      alice4business.OperationStatusPending,
		Type:           model.A4BOperationTypeResetRoom,
		Started:        time.Now(),
		Options: map[string]interface{}{
			"external_room_id": a4BRoomID,
			"partial":          partial,
		},
	})
	a.metrics.a4bOperationStarts[model.A4BOperationTypeResetRoom].Inc()
	return nil
}

func (a *Activator) startA4BRoomActivate(ctx context.Context, op *model.Operation, a4BRoomID string, withPromo bool) error {
	opID, err := a.a4b.ActivateRoom(ctx, a4BRoomID, withPromo)
	if err != nil {
		return xerrors.Errorf("unable to start activate room operation: %w", err)
	}
	a.logger.Info("Started A4B ActivateRoom operation",
		log.UInt("OperationID", op.ID),
		log.String("A4BRoomID", a4BRoomID),
		log.UInt("RoomStayID", op.RoomStayID),
		log.String("A4BOperationID", opID),
		log.String("A4BOperationType", string(model.A4BOperationTypeActivateRoom)),
		log.Bool("ApplyPromoCode", withPromo),
	)
	op.A4BBindings = append(op.A4BBindings, &model.OperationBinding{
		OperationID:    op.ID,
		Index:          len(op.A4BBindings),
		A4BOperationID: opID,
		A4BStatus:      alice4business.OperationStatusPending,
		Type:           model.A4BOperationTypeActivateRoom,
		Started:        time.Now(),
		Options: map[string]interface{}{
			"external_room_id": a4BRoomID,
			"apply_promo_code": withPromo,
		},
	})
	a.metrics.a4bOperationStarts[model.A4BOperationTypeActivateRoom].Inc()
	if withPromo {
		a.metrics.promoActivationsOnRoom.Inc()
	}
	return nil
}

func (a *Activator) updateA4BOperationBindings(tx *gorm.DB, op *model.Operation) (bool, error) {
	var hasPending bool
	for _, a4bOp := range op.A4BBindings {
		if a4bOp.A4BStatus == alice4business.OperationStatusPending {
			o, err := a.a4b.GetOperation(tx.Statement.Context, a4bOp.A4BOperationID)
			if err != nil {
				return false, xerrors.Errorf("unable to update a4b status for a4b operation binding %d: %w", a4bOp.ID, err)
			}
			if o.Status != alice4business.OperationStatusPending {
				a.logger.Info("A4B operation completed",
					log.UInt("OperationID", op.ID),
					log.String("A4BRoomID", op.RoomBinding.A4BRoomID),
					log.UInt("RoomStayID", op.RoomStayID),
					log.String("A4BOperationID", a4bOp.A4BOperationID),
					log.String("A4BOperationType", string(a4bOp.Type)),
					log.String("A4BOperationStatus", string(o.Status)))
				if o.Status == alice4business.OperationStatusRejected {
					a.metrics.a4bOperationFailures[a4bOp.Type].Inc()
				} else if o.Status == alice4business.OperationStatusResolved {
					a.metrics.a4bOperationCompletions[a4bOp.Type].Inc()
				}
				a4bOp.Finished = sql.NullTime{
					Time:  time.Now(),
					Valid: true,
				}
			}
			a4bOp.A4BStatus = o.Status
			if o.Status == alice4business.OperationStatusPending {
				hasPending = true
			}
		}
	}
	return hasPending, nil
}

func (a *Activator) notifyOperationFailed(tx *gorm.DB, op *model.Operation) {
	a.metrics.operationFailures[op.Type].Inc()
	if err := queue.Push(tx, "Activator", &events.OperationFailed{OperationId: uint64(op.ID)}); err != nil {
		a.logger.Error("Unable to notify on operation failed", log.UInt("OperationID", op.ID), log.Error(err))
	}
}

func (a *Activator) notifyOperationCancelled(tx *gorm.DB, op *model.Operation) {
	a.metrics.operationCancellations[op.Type].Inc()
	if err := queue.Push(tx, "Activator", &events.OperationCancelled{OperationId: uint64(op.ID)}); err != nil {
		a.logger.Error("Unable to notify on operation cancelled", log.UInt("OperationID", op.ID), log.Error(err))
	}
}

func (a *Activator) notifyOperationSuccess(tx *gorm.DB, op *model.Operation) {
	a.metrics.operationCompletions[op.Type].Inc()
	if err := queue.Push(tx, "Activator", &events.OperationCompleted{OperationId: uint64(op.ID)}); err != nil {
		a.logger.Error("Unable to notify on operation completion", log.UInt("OperationID", op.ID), log.Error(err))
	}
}

func New(cfg Config, pg *pgclient.PGClient, logger log.Logger, registry metrics.Registry) *Activator {
	logger = logger.WithName("Activator")
	a := &Activator{
		pg:      pg,
		logger:  logger,
		cfg:     cfg,
		queue:   queue.New("Bookings", "Activator", pg, logger, registry),
		a4b:     alice4business.NewClient(cfg.A4B),
		metrics: createActivatorMetrics(registry.WithPrefix("activator")),
	}
	a.queue.
		Subscribe(&events.RoomStayCheckedIn{}, a.onCheckIn).
		Subscribe(&events.RoomStayCheckedOut{}, a.onCheckOut).
		Subscribe(&events.RoomStayChangedRoom{}, a.onRoomChanged).
		Subscribe(&events.RoomStayCheckinCancelled{}, a.onCheckInCancelled)
	return a
}
