package activator

import (
	"time"

	"gorm.io/gorm"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
)

func (a *Activator) createOperation(tx *gorm.DB, roomStayID uint64, roomID *uint, hotelKey string, expectedStatus model.RoomStayStatus,
	operationType model.OperationType) error {
	var roomStay model.RoomStay
	if err := tx.First(&roomStay, roomStayID).Error; err != nil {
		return xerrors.Errorf("unable to get checked in room stay: %w", err)
	}
	if roomStay.Status != expectedStatus {
		a.logger.Warn("Outdated activator notification", log.UInt("RoomStayID", roomStay.ID),
			log.String("ActualStatus", string(roomStay.Status)),
			log.String("ExpectedStatus", string(expectedStatus)))
		return nil
	}
	if roomID == nil {
		roomID = roomStay.RoomID
	}

	if roomID == nil {
		a.logger.Warn("Nil room id passed", log.UInt("RoomStayID", roomStay.ID))
		return nil
	}
	var room model.Room
	if err := tx.First(&room, *roomID).Error; err != nil {
		return xerrors.Errorf("unable to get room: %w", err)
	}

	var roomBinding model.RoomBinding
	if err := tx.Where(model.RoomBinding{RoomID: room.ID, ActivatorEnabled: true}).First(&roomBinding).Error; err != nil {
		if xerrors.Is(err, gorm.ErrRecordNotFound) {
			a.logger.Debug("No room binding",
				log.UInt("RoomStayID", roomStay.ID),
				log.String("RoomName", room.Name),
				log.String("HotelKey", hotelKey))
			return nil
		}
		return xerrors.Errorf("unable to get room in room stay: %w", err)
	}
	if err := a.cancelOtherOperations(tx, &roomBinding); err != nil {
		return err
	}
	op := model.Operation{
		Type:          operationType,
		State:         model.OperationStateNew,
		RoomBindingID: roomBinding.ID,
		RoomStayID:    roomStay.ID,
		CheckAfter:    time.Now(),
	}
	if err := tx.Create(&op).Error; err != nil {
		return xerrors.Errorf("unable to create operation of type %s: %w", operationType, err)
	}
	a.metrics.operationStarts[op.Type].Inc()
	a.logger.Info("Operation created",
		log.UInt("OperationID", op.ID),
		log.String("HotelKey", hotelKey),
		log.String("RoomName", room.Name),
		log.String("OperationType", string(operationType)))
	return nil
}

func (a *Activator) getUpcomingStayForRoom(tx *gorm.DB, rb *model.RoomBinding) *model.RoomStay {
	now := time.Now()
	today := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, time.Local)
	var roomStay model.RoomStay
	if err := tx.Model(model.RoomStay{}).
		Where(model.RoomStay{
			RoomID: &rb.RoomID,
			Status: model.RoomStayStatusNew,
		}).
		Where("check_in_date_time > ?", today).
		Order("check_in_date_time").
		First(&roomStay).Error; err != nil {
		if xerrors.Is(err, gorm.ErrRecordNotFound) {
			return nil
		}
		a.logger.Warn("Unable to fetch next room stay of the room",
			log.UInt("RoomID", rb.RoomID),
			log.Error(err))

	}
	return &roomStay
}

func (a *Activator) cancelOtherOperations(tx *gorm.DB, rb *model.RoomBinding) error {
	var opsToCancel []model.Operation
	if err := tx.Model(model.Operation{}).
		Where(model.Operation{State: model.OperationStateNew, RoomBindingID: rb.ID}).
		Find(&opsToCancel).Error; err == nil {
		for _, op := range opsToCancel {
			a.metrics.operationCancellations[op.Type].Inc()
		}
	}
	updates := tx.Model(model.Operation{}).
		Where(model.Operation{State: model.OperationStateNew, RoomBindingID: rb.ID}).
		Updates(model.Operation{State: model.OperationStateCancelled})
	if err := updates.Error; err != nil {
		return xerrors.Errorf("unable to cancel operations in state new: %w", err)
	}

	if err := tx.Model(model.Operation{}).
		Where(model.Operation{State: model.OperationStateRunning, RoomBindingID: rb.ID}).
		Updates(model.Operation{State: model.OperationStateCancelling}).Error; err != nil {
		return xerrors.Errorf("unable to cancel operations in state running: %w", err)
	}
	return nil
}
