package alice4business

import (
	"context"
	"fmt"
	"io"
	"net/url"
	"time"

	"github.com/go-resty/resty/v2"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type Client struct {
	config     Config
	httpClient *resty.Client
}

const ok = "ok"
const a4bTimeFormat = "2006-01-02 15:04:05"

func (c *Client) GetRoom(ctx context.Context, externalID string) (*RoomInfo, error) {
	var result roomInfoResponse
	if err := c.doExecute(ctx, resty.MethodGet, c.getURL("/api/public/rooms/info"), nil, url.Values{"external_room_id": {externalID}}, &result); err != nil {
		return nil, err
	}
	if result.Status != ok {
		return nil, xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	return &result.Info, nil
}

func (c *Client) GetDevice(ctx context.Context, externalID string) (*DeviceInfo, error) {
	var result deviceInfoResponse
	if err := c.doExecute(ctx, resty.MethodGet, c.getURL("/api/public/devices/info"), nil, url.Values{"external_id": {externalID}}, &result); err != nil {
		return nil, err
	}
	if result.Status != ok {
		return nil, xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	return &result.Info, nil
}

func (c *Client) ResetRoom(ctx context.Context, externalID string, partial bool) (string, error) {
	var result startOperationResponse
	if err := c.doExecute(ctx, resty.MethodPost, c.getURL("/api/public/rooms/reset"), map[string]interface{}{
		"external_room_id": externalID,
		"partial":          partial,
	}, nil, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.OperationID, nil
}

func (c *Client) ActivateRoom(ctx context.Context, externalID string, applyPromoCode bool) (string, error) {
	var result startOperationResponse
	if err := c.doExecute(ctx, resty.MethodPost, c.getURL("/api/public/rooms/activate"), map[string]interface{}{
		"external_room_id": externalID,
		"with_promocode":   applyPromoCode,
	}, nil, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.OperationID, nil
}

func (c *Client) ActivateDevice(ctx context.Context, externalID string, applyPromoCode bool) (string, error) {
	var result startOperationResponse
	if err := c.doExecute(ctx, resty.MethodPost, c.getURL("/api/public/devices/activate"), map[string]interface{}{
		"external_id":    externalID,
		"with_promocode": applyPromoCode,
	}, nil, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.OperationID, nil
}

func (c *Client) ResetDevice(ctx context.Context, externalID string) (string, error) {
	var result startOperationResponse
	if err := c.doExecute(ctx, resty.MethodPost, c.getURL("/api/public/devices/reset"), map[string]string{"external_id": externalID}, nil, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.OperationID, nil
}

func (c *Client) CreateRoomActivation(ctx context.Context, externalID string, firstActivation *time.Time, lastActivation *time.Time) (string, error) {
	if firstActivation == nil || lastActivation == nil {
		return "", fmt.Errorf("activation time boundaries must be set")
	}
	payload := map[string]string{
		"external_id":           externalID,
		"first_activation_date": firstActivation.Format(a4bTimeFormat),
		"last_activation_date":  lastActivation.Format(a4bTimeFormat),
	}
	var result activationResponse
	if err := c.doExecute(ctx, resty.MethodPost, c.getURL("/api/public/rooms/activation"), payload, nil, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.ID, nil
}

func (c *Client) CreateDeviceActivation(ctx context.Context, externalID string, checkin time.Time, checkout time.Time) (string, error) {
	payload := map[string]string{
		"external_id":           externalID,
		"first_activation_date": checkin.Format(a4bTimeFormat),
		"last_activation_date":  checkout.Format(a4bTimeFormat),
	}
	var result activationResponse
	if err := c.doExecute(ctx, resty.MethodPost, c.getURL("/api/public/devices/activation"), payload, nil, &result); err != nil {
		return "", err
	}
	if result.Status != ok {
		return "", xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	return result.ID, nil
}

func (c *Client) GetOperation(ctx context.Context, operationID string) (*Operation, error) {
	var result getOperationResponse
	if err := c.doExecute(ctx, resty.MethodGet, c.getURL("/api/public/operations"), nil, url.Values{"id": {operationID}}, &result); err != nil {
		return nil, err
	}
	if result.Status != ok {
		return nil, xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	op := result.Operation
	op.ID = operationID
	return &op, nil
}

func (c *Client) ApplyPromoForDevice(ctx context.Context, externalID string) error {
	var result baseResponse
	payload := map[string]string{"external_id": externalID}
	if err := c.doExecute(ctx, resty.MethodPost, c.getURL("/api/public/devices/promocode"), payload, nil, &result); err != nil {
		return err
	}
	if result.Status != ok {
		return xerrors.Errorf("unexpected status in status response: %s", result.Status)
	}
	return nil
}

func (c *Client) doExecute(ctx context.Context, method, path string, body interface{}, queryParams url.Values, result interface{}) error {
	var errResponse baseResponse
	r := c.httpClient.R().
		SetContext(ctx).
		SetAuthToken(c.config.OAuthToken).
		SetAuthScheme("OAuth").
		SetBody(body).
		SetError(&errResponse)
	if result != nil {
		r = r.SetResult(result)
	}
	if queryParams != nil {
		r = r.SetQueryParamsFromValues(queryParams)
	}
	response, err := r.Execute(method, path)
	if err != nil {
		return ResponseError.Wrap(err)
	}
	if !response.IsSuccess() {
		if errResponse.Error != nil {
			return xerrors.Errorf("error response from a4b service: %w", errResponse.Error)
		} else {
			raw, _ := io.ReadAll(response.RawResponse.Body)
			return xerrors.Errorf("unexpected response from a4b service: %v", raw)
		}
	}
	return nil
}

func (c *Client) getURL(endpoint string) string {
	return fmt.Sprintf("%s%s", c.config.APIEndpoint, endpoint)
}

func NewClient(cfg Config) *Client {
	return &Client{config: cfg,
		httpClient: resty.New().SetTimeout(cfg.Timeout)}
}
