package app

import (
	"fmt"

	"golang.yandex/hasql"
	gormlogger "gorm.io/gorm/logger"

	"a.yandex-team.ru/library/go/core/log"
	arczap "a.yandex-team.ru/library/go/core/log/zap"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/pgclient"
)

func BuildPGClient(cfg DBConfig) (*pgclient.PGClient, error) {
	logLevel := gormlogger.Silent
	if cfg.ShowSQL {
		logLevel = gormlogger.Info
	}

	return pgclient.NewPGClient(
		cfg.Hosts,
		cfg.Port,
		cfg.Name,
		cfg.User,
		cfg.Password,
		pgclient.DefaultInitTimeout,
		[]pgclient.ClientOption{pgclient.WithLogLevel(logLevel)},
		append(pgclient.DefaultClusterOptions, hasql.WithUpdateTimeout(cfg.HostsUpdateTimeout)),
	)
}

func NewLogger(cfg LogConfig) (log.Logger, error) {
	level, err := log.ParseLevel(cfg.Level)
	if err != nil {
		return nil, fmt.Errorf("bad log level format: %w", err)
	}
	var logger log.Logger
	switch cfg.Type {
	case consoleLogType:
		logger, err = arczap.New(arczap.ConsoleConfig(level))
	case deployLogType:
		logger, err = arczap.NewDeployLogger(level)
	default:
		err = xerrors.Errorf("bad log type '%s'", cfg.Type)
	}
	if err != nil {
		return nil, xerrors.Errorf("unable to create logger: %w", err)
	}
	return logger, nil
}
