package app

import (
	"time"

	metricsserver "a.yandex-team.ru/travel/library/go/metrics/server"
)

type EnvType string

const developmentEnv EnvType = "development"
const testingEnv EnvType = "testing"
const productionEnv EnvType = "production"

type Config struct {
	EnvType EnvType `config:"yenv-type,required" yaml:"yandex_environment_type"`
	DB      DBConfig
	Logging LogConfig
	Metrics metricsserver.MetricsConfig
}

type LogType string

const consoleLogType LogType = "console"
const deployLogType LogType = "deploy"

type LogConfig struct {
	Type  LogType `config:"logging-type" yaml:"type"`
	Level string  `config:"logging-level" yaml:"logging_level"`
}

type DBConfig struct {
	Hosts              []string      `config:"DATABASE_HOSTS,required" yaml:"hosts"`
	Port               int           `config:"DATABASE_PORT,required" yaml:"port"`
	User               string        `config:"DATABASE_USER,required" yaml:"user"`
	Password           string        `config:"DATABASE_PASSWORD,required" yaml:"password"`
	Name               string        `config:"DATABASE_NAME,required" yaml:"name"`
	HostsUpdateTimeout time.Duration `yaml:"hosts_update_timeout"`
	MigrateOnStart     bool          `yaml:"migrate_on_start"`
	ShowSQL            bool          `config:"SHOW_DB_SQL" yaml:"show_sql"`
}

var DefaultConfig = Config{
	EnvType: developmentEnv,
	DB: DBConfig{
		Hosts:              []string{"localhost"},
		Port:               5432,
		Name:               "metapms",
		User:               "metapms",
		HostsUpdateTimeout: 5 * time.Second,
	},
	Logging: LogConfig{
		Type:  consoleLogType,
		Level: "info",
	},
	Metrics: metricsserver.DefaultMetricsConfig,
}

func (c Config) IsDevelopment() bool {
	return c.EnvType == "development"
}
