package model

import (
	"context"
	"database/sql"
	"database/sql/driver"
	"encoding/json"
	"fmt"
	"strconv"
	"strings"
	"time"

	"gorm.io/gorm"
	"gorm.io/gorm/clause"
	"gorm.io/gorm/schema"

	"a.yandex-team.ru/library/go/core/xerrors"
)

type Date time.Time
type CommaStringList []string
type CommaIntList []int
type JSONMap map[string]interface{}

func (date *Date) Scan(value interface{}) (err error) {
	nullTime := &sql.NullTime{}
	err = nullTime.Scan(value)
	*date = Date(nullTime.Time)
	return
}

func (date Date) Value() (driver.Value, error) {
	y, m, d := time.Time(date).Date()
	return time.Date(y, m, d, 0, 0, 0, 0, time.Time(date).Location()), nil
}

func (date Date) GormDataType() string {
	return "date"
}

func (date Date) GobEncode() ([]byte, error) {
	return time.Time(date).GobEncode()
}

func (date *Date) GobDecode(b []byte) error {
	return (*time.Time)(date).GobDecode(b)
}

func (date Date) MarshalJSON() ([]byte, error) {
	return time.Time(date).MarshalJSON()
}

func (date *Date) UnmarshalJSON(b []byte) error {
	return (*time.Time)(date).UnmarshalJSON(b)
}

func (l *CommaStringList) Scan(value interface{}) (err error) {
	if value == nil {
		*l = CommaStringList{}
		return nil
	}
	var str string
	switch v := value.(type) {
	case []byte:
		str = string(v)
	case string:
		str = v
	default:
		return xerrors.Errorf("failed to unmarshal comma-string-list value %v", value)
	}
	if str == "" {
		*l = CommaStringList{}
		return nil
	}
	*l = strings.Split(str, ",")
	return nil
}

func (l CommaStringList) Value() (driver.Value, error) {
	res := strings.Join(l, ",")
	return res, nil
}

func (l CommaStringList) GormDataType() string {
	return "text"
}

func (l *CommaIntList) Scan(value interface{}) (err error) {
	if value == nil {
		*l = CommaIntList{}
		return nil
	}
	var str string
	switch v := value.(type) {
	case []byte:
		str = string(v)
	case string:
		str = v
	default:
		return xerrors.Errorf("failed to unmarshal comma-int-list value %v", value)
	}
	if str == "" {
		*l = CommaIntList{}
		return nil
	}
	stringValues := strings.Split(str, ",")
	res := make([]int, len(stringValues))
	for i, v := range stringValues {
		vv, err := strconv.Atoi(v)
		if err != nil {
			return err
		}
		res[i] = vv
	}
	*l = res
	return nil
}

func (l CommaIntList) Value() (driver.Value, error) {
	stringValues := make([]string, len(l))
	for i, v := range l {
		stringValues[i] = strconv.Itoa(v)
	}
	res := strings.Join(stringValues, ",")
	return res, nil
}

func (l CommaIntList) GormDataType() string {
	return "text"
}

func (m JSONMap) Value() (driver.Value, error) {
	if m == nil {
		return nil, nil
	}
	ba, err := m.MarshalJSON()
	return string(ba), err
}

func (m *JSONMap) Scan(val interface{}) error {
	if val == nil {
		*m = make(JSONMap)
		return nil
	}
	var ba []byte
	switch v := val.(type) {
	case []byte:
		ba = v
	case string:
		ba = []byte(v)
	default:
		return xerrors.New(fmt.Sprint("Failed to unmarshal JSONB value:", val))
	}
	t := map[string]interface{}{}
	err := json.Unmarshal(ba, &t)
	*m = t
	return err
}

func (m JSONMap) MarshalJSON() ([]byte, error) {
	if m == nil {
		return []byte("null"), nil
	}
	t := (map[string]interface{})(m)
	return json.Marshal(t)
}

func (m *JSONMap) UnmarshalJSON(b []byte) error {
	t := map[string]interface{}{}
	err := json.Unmarshal(b, &t)
	*m = t
	return err
}

func (m JSONMap) GormDataType() string {
	return "jsonmap"
}

func (m JSONMap) GormDBDataType(db *gorm.DB, field *schema.Field) string {
	return "JSONB"
}

func (m JSONMap) GormValue(ctx context.Context, db *gorm.DB) clause.Expr {
	data, _ := m.MarshalJSON()
	return gorm.Expr("?", string(data))
}
