package model

import (
	"gorm.io/gorm"
)

const changeCallbacksAttr = "change-callbacks-attr"

type ChangeCallback interface {
	OnNewEntity(tx *gorm.DB, entity interface{}) error
	OnSavedEntity(tx *gorm.DB, entity interface{}) error
	OnFieldsChanged(tx *gorm.DB, entity interface{}, fields []string) error
}

func WithChangeCallback(tx *gorm.DB, callback ChangeCallback) *gorm.DB {
	callbacksInt, exists := tx.Get(changeCallbacksAttr)
	var callbacks []ChangeCallback
	if !exists {
		callbacks = []ChangeCallback{callback}
	} else {
		callbacks = callbacksInt.([]ChangeCallback)
		callbacks = append(callbacks, callback)
	}
	return tx.Set(changeCallbacksAttr, callbacks)
}

func getCallbacks(tx *gorm.DB) []ChangeCallback {
	callbacksInt, exists := tx.Get(changeCallbacksAttr)
	if !exists {
		return nil
	}
	return callbacksInt.([]ChangeCallback)
}

func (b *Booking) AfterCreate(tx *gorm.DB) error {
	for _, cb := range getCallbacks(tx) {
		if err := cb.OnNewEntity(tx, b); err != nil {
			return err
		}
	}
	return nil
}

func (b *Booking) BeforeUpdate(tx *gorm.DB) error {
	return onFieldsChanged(tx, b, "CustomerLanguage", "VisitPurpose", "CustomerComment", "LastModifiedAt",
		"GroupName", "CurrencyID", "GuestID", "Source", "SourceChannelName")
}

func (b *Booking) AfterUpdate(tx *gorm.DB) error {
	return onUpdate(tx, b)
}

func (rs *RoomStay) AfterCreate(tx *gorm.DB) error {
	for _, cb := range getCallbacks(tx) {
		if err := cb.OnNewEntity(tx, rs); err != nil {
			return err
		}
	}
	return nil
}

func (rs *RoomStay) BeforeUpdate(tx *gorm.DB) error {
	return onFieldsChanged(tx, rs, "RoomID", "CheckInDateTime", "CheckOutDateTime", "ActualCheckInDateTime",
		"ActualCheckOutDateTime", "Status", "BookingStatus", "GuestCountInfo", "DepositAmount", "DepositCurrency",
		"TotalAmount", "ToRefundAmount", "ToPayAmount", "Amenities")
}

func (rs *RoomStay) AfterUpdate(tx *gorm.DB) error {
	return onUpdate(tx, rs)
}

func onFieldsChanged(tx *gorm.DB, entity interface{}, fieldsToCheck ...string) error {
	callbacks := getCallbacks(tx)
	if len(callbacks) == 0 {
		return nil
	}
	var changedFields []string
	for _, f := range fieldsToCheck {
		if tx.Statement.Changed(f) {
			changedFields = append(changedFields, f)
		}
	}
	if len(changedFields) > 0 {
		for _, cb := range callbacks {
			if err := cb.OnFieldsChanged(tx, entity, changedFields); err != nil {
				return err
			}
		}
	}
	return nil
}

func onUpdate(tx *gorm.DB, entity interface{}) error {
	callbacks := getCallbacks(tx)
	if len(callbacks) == 0 {
		return nil
	}
	for _, cb := range callbacks {
		if err := cb.OnSavedEntity(tx, entity); err != nil {
			return err
		}
	}
	return nil
}
