package model

import (
	"database/sql"

	"gorm.io/gorm"
)

type RoomStayStatus string
type BookingStatus string

const (
	RoomStayStatusNew        RoomStayStatus = "New"
	RoomStayStatusCheckedIn  RoomStayStatus = "CheckedIn"
	RoomStayStatusCheckedOut RoomStayStatus = "CheckedOut"
	RoomStayStatusCancelled  RoomStayStatus = "Cancelled"
)

const (
	BookingStatusUnconfirmed BookingStatus = "Unconfirmed"
	BookingStatusConfirmed   BookingStatus = "Confirmed"
	BookingStatusCancelled   BookingStatus = "Cancelled"
	BookingStatusReleased    BookingStatus = "Released"
	BookingStatusPending     BookingStatus = "Pending"
	BookingStatusEdited      BookingStatus = "Edited"
)

type Booking struct {
	gorm.Model
	HotelID           uint
	TravellineID      string `gorm:"uniqueIndex;not null;<-:create"`
	TravellineNumber  string `gorm:"not null"`
	CustomerLanguage  string
	VisitPurpose      JSONMap
	CustomerComment   string
	LastModifiedAt    sql.NullTime
	GroupName         string
	CurrencyID        string
	GuestID           uint
	RoomStays         []*RoomStay
	Guest             *Guest
	Hotel             *Hotel
	Notifications     []*Notification
	Source            JSONMap
	SourceChannelName string
}

type Guest struct {
	gorm.Model
	TravellineID    string `gorm:"uniqueIndex;not null;<-:create"`
	LastName        string
	FirstName       string
	MiddleName      string
	BirthDate       *Date
	CitizenshipCode string
	Status          JSONMap
	Emails          CommaStringList
	Phones          CommaStringList
	Gender          string
}

type RoomStay struct {
	gorm.Model
	TravellineID           string `gorm:"uniqueIndex;not null;<-:create"`
	BookingID              uint
	RoomID                 *uint
	Room                   *Room
	RoomTLID               string
	RoomTypeTLID           string
	CheckInDateTime        sql.NullTime
	CheckOutDateTime       sql.NullTime
	ActualCheckInDateTime  sql.NullTime
	ActualCheckOutDateTime sql.NullTime
	Status                 RoomStayStatus
	BookingStatus          BookingStatus
	Booking                *Booking
	GuestCountInfo         *GuestCountInfo `gorm:"embedded;embeddedPrefix:guest_count_info_"`
	Deposit                *Amount         `gorm:"embedded;embeddedPrefix:deposit_"`
	TotalPrice             *TotalPrice     `gorm:"embedded;embeddedPrefix:total_price_"`
	Amenities              CommaStringList
}

type Room struct {
	gorm.Model
	TravellineID   string `gorm:"uniqueIndex;not null;<-:create"`
	Name           string
	RoomTypeTLID   string
	HotelID        uint
	Hotel          *Hotel
	RoomBinding    *RoomBinding
	ReportsEnabled sql.NullBool `gorm:"default:true"`
}

// embedded structs

type GuestCountInfo struct {
	Adults   int
	Children int
}

type Amount struct {
	DepositAmount   float64
	DepositCurrency string
}

type TotalPrice struct {
	TotalAmount    float64
	ToPayAmount    float64
	ToRefundAmount float64
}
