package notifier

import (
	"time"

	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
)

type TimeHM struct {
	Hour   int
	Minute int
}

type Config struct {
	PeriodicCheckInterval         time.Duration
	PeriodicCheckBookingFreshness time.Duration
	ImmediatePollInterval         time.Duration
	MinPromoValidity              time.Duration
	Enabled                       bool
}

var DefaultConfig = Config{
	PeriodicCheckBookingFreshness: 60 * time.Hour * 24,
	PeriodicCheckInterval:         1 * time.Hour,
	ImmediatePollInterval:         1 * time.Second,
	Enabled:                       true,
	MinPromoValidity:              time.Hour * 24 * 30,
}

type NotificationRule struct {
	NotificationType model.NotificationType
	FilterProviders  []RoomStayFilterProvider
	PayloadBuilder   PayloadBuilder
	StartTime        *TimeHM
	EndTime          *TimeHM
}

// Event-based notifications
var newBookingNotification = NotificationRule{
	NotificationType: model.NotificationTypeConfirmedBooking,
	PayloadBuilder:   stayDatesPayloadBuilder,
}

var checkedInNotification = NotificationRule{
	NotificationType: model.NotificationTypeOnCheckin,
	PayloadBuilder:   checkinPayloadBuilder,
}

var bookingCancelledNotification = NotificationRule{
	NotificationType: model.NotificationTypeCancelledStay,
	PayloadBuilder:   cancellationPayloadBuilder,
}

var periodicNotifications = []NotificationRule{
	{
		NotificationType: model.NotificationTypePreCheckin,
		FilterProviders: []RoomStayFilterProvider{
			StayStatusIs(model.RoomStayStatusNew),
			StayCheckInAlreadyStartedOrStartsInDuration(time.Hour * 26),
			StayCheckOutInFuture(),
			BookingStatusIs(model.BookingStatusConfirmed),
		},
		StartTime:      &TimeHM{Hour: 10, Minute: 0},
		EndTime:        &TimeHM{Hour: 21, Minute: 0},
		PayloadBuilder: nil,
	},
	{
		NotificationType: model.NotificationTypeInStay,
		FilterProviders: []RoomStayFilterProvider{
			StayStatusIs(model.RoomStayStatusCheckedIn),
			StayActualCheckInWasMoreThanDurationAgo(time.Hour * 24),
			StayLastsAtLeastDays(2),
			StayCheckOutInFuture(),
		},
		StartTime:      &TimeHM{Hour: 10, Minute: 0},
		EndTime:        &TimeHM{Hour: 21, Minute: 0},
		PayloadBuilder: marketPayloadBuilder,
	},
	{
		NotificationType: model.NotificationTypePostStay,
		FilterProviders: []RoomStayFilterProvider{
			StayStatusIs(model.RoomStayStatusCheckedOut),
			StayActualCheckOutWasMoreThanDurationAgo(time.Hour * 24),
			StayActualCheckOutWasLessThanDurationAgo(time.Hour * 24 * 7), // will not notify bookings ended more than 2 weeks from now
		},
		StartTime:      &TimeHM{Hour: 10, Minute: 0},
		EndTime:        &TimeHM{Hour: 21, Minute: 0},
		PayloadBuilder: nil,
	},
	{
		NotificationType: model.NotificationTypePostStayAskona,
		FilterProviders: []RoomStayFilterProvider{
			StayStatusIs(model.RoomStayStatusCheckedOut),
			StayActualCheckOutWasMoreThanDurationAgo(time.Hour * 25),
			StayActualCheckOutWasLessThanDurationAgo(time.Hour * 24 * 7), // will not notify bookings ended more than 2 weeks from now
		},
		StartTime:      &TimeHM{Hour: 10, Minute: 0},
		EndTime:        &TimeHM{Hour: 21, Minute: 0},
		PayloadBuilder: marketPayloadBuilder,
	},
}

func (r *NotificationRule) IsActiveForHotel(hotel *model.Hotel) (bool, error) {
	location, err := hotel.GetLocation()
	if err != nil {
		return false, err
	}
	now := time.Now().In(location)
	start := time.Date(now.Year(), now.Month(), now.Day(), r.StartTime.Hour, r.StartTime.Minute, 0, 0, location)
	end := time.Date(now.Year(), now.Month(), now.Day(), r.EndTime.Hour, r.EndTime.Minute, 0, 0, location)
	return now.After(start) && end.After(now), nil
}
