package notifier

import (
	"time"

	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
)

type roomStayFilter struct {
	field string
	op    string
	value interface{}
}

type RoomStayFilterProvider func() []roomStayFilter

func BookingStatusIs(status model.BookingStatus) RoomStayFilterProvider {
	return func() []roomStayFilter {
		return []roomStayFilter{
			{
				field: "rs.booking_status",
				op:    "=",
				value: status,
			},
		}
	}
}

func StayStatusIs(status model.RoomStayStatus) RoomStayFilterProvider {
	return func() []roomStayFilter {
		return []roomStayFilter{
			{
				field: "rs.status",
				op:    "=",
				value: status,
			},
		}
	}
}

func StayStatusIn(statuses ...model.RoomStayStatus) RoomStayFilterProvider {
	return func() []roomStayFilter {
		return []roomStayFilter{
			{
				field: "rs.status",
				op:    "IN",
				value: statuses,
			},
		}
	}
}

func StayCheckInInFuture() RoomStayFilterProvider {
	return func() []roomStayFilter {
		now := time.Now()
		return []roomStayFilter{
			{
				field: "rs.check_in_date_time",
				op:    ">",
				value: now,
			},
		}
	}
}

func StayLastsAtLeastDays(days int) RoomStayFilterProvider {
	return func() []roomStayFilter {
		return []roomStayFilter{
			{
				field: "rs.check_out_date_time::timestamp::date - rs.check_in_date_time::timestamp::date",
				op:    ">",
				value: days,
			},
		}
	}
}

func StayCheckOutInFuture() RoomStayFilterProvider {
	return func() []roomStayFilter {
		now := time.Now()
		return []roomStayFilter{
			{
				field: "rs.check_out_date_time",
				op:    ">=",
				value: now,
			},
		}
	}
}

func StayCheckInAlreadyStartedOrStartsInDuration(d time.Duration) RoomStayFilterProvider {
	return func() []roomStayFilter {
		now := time.Now()
		return []roomStayFilter{
			{
				field: "rs.check_in_date_time",
				op:    "<",
				value: now.Add(d),
			},
		}
	}
}

func StayActualCheckInWasMoreThanDurationAgo(d time.Duration) RoomStayFilterProvider {
	return func() []roomStayFilter {
		now := time.Now()
		return []roomStayFilter{
			{
				field: "rs.actual_check_in_date_time",
				op:    "<",
				value: now.Add(-1 * d),
			},
		}
	}
}

func StayActualCheckOutWasMoreThanDurationAgo(d time.Duration) RoomStayFilterProvider {
	return func() []roomStayFilter {
		now := time.Now()
		return []roomStayFilter{
			{
				field: "rs.actual_check_out_date_time",
				op:    "<",
				value: now.Add(-1 * d),
			},
		}
	}
}

func StayActualCheckOutWasLessThanDurationAgo(d time.Duration) RoomStayFilterProvider {
	return func() []roomStayFilter {
		now := time.Now()
		return []roomStayFilter{
			{
				field: "rs.actual_check_out_date_time",
				op:    ">",
				value: now.Add(-1 * d),
			},
		}
	}
}
