package notifier

import (
	"context"
	"fmt"
	"strings"
	"sync"
	"time"

	"google.golang.org/protobuf/proto"
	"gorm.io/gorm"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/events"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
	"a.yandex-team.ru/travel/budapest/metapms/internal/pgclient"
	"a.yandex-team.ru/travel/budapest/metapms/internal/queue"
)

const periodicNotifierLock = 1001

type Notifier struct {
	pg     *pgclient.PGClient
	logger log.Logger
	cfg    Config
	queue  *queue.Queue
}

func New(cfg Config, pg *pgclient.PGClient, logger log.Logger, registry metrics.Registry) *Notifier {
	logger = logger.WithName("Notifier")
	n := &Notifier{
		logger: logger,
		cfg:    cfg,
		pg:     pg,
		queue:  queue.New("Bookings", "ImmediateNotifier", pg, logger, registry),
	}
	n.queue.
		Subscribe(&events.BookingAdded{}, n.onBookingAdded).
		Subscribe(&events.BookingConfirmed{}, n.onBookingConfirmed).
		Subscribe(&events.RoomStayCheckedIn{}, n.onRoomStayCheckedIn).
		Subscribe(&events.BookingCancelled{}, n.onBookingCancelled)
	return n
}

type PayloadBuilder func(tx *gorm.DB, cfg *Config, notification *model.Notification) error

func findBookingsWithNoNotificationsOfType(tx *gorm.DB, notificationType model.NotificationType, freshnessWindow time.Duration, filterProviders ...RoomStayFilterProvider) ([]*model.Booking, error) {
	baseQuery := `
SELECT DISTINCT b.ID
FROM bookings b
LEFT JOIN (
	SELECT booking_id
	FROM notifications
	WHERE type = ? AND state in ?
) n
ON n.booking_id = b.id
INNER JOIN room_stays rs
on rs.booking_id = b.id
WHERE n IS null AND b.last_modified_at > ?
`
	var roomStayFilters []roomStayFilter
	for _, prov := range filterProviders {
		roomStayFilters = append(roomStayFilters, prov()...)
	}
	params := make([]interface{}, len(roomStayFilters)+3)
	params[0] = notificationType
	params[1] = model.NotificationStatesProcessed
	params[2] = time.Now().Add(-1 * freshnessWindow)
	extension := ""
	for i, cond := range roomStayFilters {
		extension = fmt.Sprintf("%s AND %s %s ?", extension, cond.field, cond.op)
		params[3+i] = cond.value
	}
	query := baseQuery + extension
	var bookingIds []uint
	if err := tx.Raw(query, params...).Scan(&bookingIds).Error; err != nil {
		return nil, xerrors.Errorf("unable to find booking ids to notify: %w", err)
	}
	if len(bookingIds) == 0 {
		return nil, nil
	}
	var bookings []*model.Booking
	if err := tx.Preload("RoomStays.Room").
		Preload("Notifications").
		Preload("Guest").
		Preload("Hotel").
		Find(&bookings, bookingIds).Error; err != nil {
		return nil, xerrors.Errorf("unable to load bookings to notify: %w", err)
	}
	return bookings, nil
}

func (n *Notifier) Run(ctx context.Context) error {
	if !n.cfg.Enabled {
		return nil
	}
	if err := n.periodicNotify(); err != nil {
		if !pgclient.IsLockError(err) {
			return xerrors.Errorf("unable to run initial notifications: %w", err)
		} else {
			n.logger.Warn("Lock conflict on initial notification run")
		}
	}
	wg := sync.WaitGroup{}
	wg.Add(2)
	go func() {
		defer wg.Done()
		err := n.queue.Listen(ctx, n.cfg.ImmediatePollInterval)
		if err != nil {
			n.logger.Error("error in event listening loop", log.Error(err))
		}
	}()
	go func() {
		defer wg.Done()
		err := n.runPeriodicChecks(ctx)
		if err != nil {
			n.logger.Error("error in periodic check loop", log.Error(err))
		}
	}()
	wg.Wait()
	return nil
}

func (n *Notifier) runPeriodicChecks(ctx context.Context) error {
	ticker := time.NewTicker(n.cfg.PeriodicCheckInterval)
	for {
		select {
		case <-ticker.C:
			err := n.periodicNotify()
			if err != nil {
				if !pgclient.IsLockError(err) {
					n.logger.Error("Error while running periodic notifications", log.Error(err))
				}
				continue
			}
		case <-ctx.Done():
			return nil
		}
	}
}

func (n *Notifier) periodicNotify() error {
	return pgclient.WithLock(context.Background(), n.pg, n.logger, periodicNotifierLock, func(ctx context.Context, tx *gorm.DB) error {
		n.logger.Debug("PeriodicNotifications::Begin")
		defer n.logger.Debug("PeriodicNotifications::End")
		for _, rule := range periodicNotifications {
			if err := n.periodicNotifyRule(tx, rule); err != nil {
				return err
			}
		}
		return nil
	})
}

func (n *Notifier) periodicNotifyRule(db *gorm.DB, rule NotificationRule) error {
	n.logger.Debug("Notification::Begin", log.String("NotificationType", string(rule.NotificationType)))
	defer n.logger.Debug("Notification::End", log.String("NotificationType", string(rule.NotificationType)))
	err := db.Transaction(func(tx *gorm.DB) error {
		bookings, err := findBookingsWithNoNotificationsOfType(tx, rule.NotificationType, n.cfg.PeriodicCheckBookingFreshness, rule.FilterProviders...)
		if err != nil {
			return xerrors.Errorf("unable to search for bookings to notify: %w", err)
		}
		for _, b := range bookings {
			active, err := rule.IsActiveForHotel(b.Hotel)
			if err != nil {
				n.logger.Error("Error while checking rule activity",
					log.String("NotificationType", string(rule.NotificationType)),
					log.UInt("HotelID", b.HotelID),
					log.Error(err))
				continue
			}
			if !active {
				continue
			}
			notification := n.createNotificationEntity(tx, rule, b)
			if err := tx.Omit("Booking").Create(&notification).Error; err != nil {
				return xerrors.Errorf("unable to create new notification: %w", err)
			} else {
				n.logNewNotification(&notification)
			}
		}
		return nil
	})
	if err != nil {
		n.logger.Error("Unable to send notifications", log.String("NotificationType", string(rule.NotificationType)), log.Error(err))
		return err
	}
	return nil
}

func (n *Notifier) createNotificationEntity(tx *gorm.DB, rule NotificationRule, b *model.Booking) model.Notification {
	notification := model.Notification{
		Type:      rule.NotificationType,
		BookingID: b.ID,
		Booking:   b,
	}
	var payloadErr error
	if rule.PayloadBuilder != nil {
		payloadErr = rule.PayloadBuilder(tx, &n.cfg, &notification)
	}
	if payloadErr != nil {
		n.logger.Error("Unable to generate payload", log.String("NotificationType", string(rule.NotificationType)), log.Error(payloadErr))
		notification.State = model.NotificationStateError
	} else {
		if notification.TemplateKey == "" {
			notification.TemplateKey = fmt.Sprintf("%s_%s", strings.ToLower(notification.Booking.Hotel.Key), strings.ToLower(string(notification.Type)))
		}
		if len(notification.BoundPromos) > 0 {
			notification.TemplateKey += "_promo"
		}
		if len(b.Guest.Phones) > 0 {
			notifyErr := n.sendNotification(b.Guest.Phones[0], &notification)
			if notifyErr != nil {
				n.logger.Error("Unable to send notification", log.String("NotificationType", string(rule.NotificationType)), log.Error(notifyErr))
				notification.State = model.NotificationStateError
			} else {
				notification.State = model.NotificationStateSent
			}
		} else {
			notification.State = model.NotificationStateSkipped
		}
	}
	notification.ProcessedAt = time.Now()
	return notification
}

func (n *Notifier) logNewNotification(notification *model.Notification) {
	n.logger.Info("Notification created",
		getNotificationLogFields(notification)...,
	)
}

func (n *Notifier) sendNotification(phoneNumber string, notification *model.Notification) error {
	// TODO: actual sending here
	n.logger.Warn("Mocked notification",
		append(getNotificationLogFields(notification),
			log.String("Addressee", phoneNumber),
			log.String("Template", notification.TemplateKey),
			log.Any("Payload", notification.Payload))...,
	)
	return nil
}

func (n *Notifier) createNotificationOfType(tx *gorm.DB, booking *model.Booking, rule NotificationRule) error {
	for _, ntf := range booking.Notifications {
		if ntf.Type == rule.NotificationType {
			ntf.Booking = booking
			n.logger.Warn("Notification of this type already exists for this booking", getNotificationLogFields(ntf)...)
			return nil
		}
	}
	notification := n.createNotificationEntity(tx, rule, booking)
	if err := tx.Omit("Booking").Create(&notification).Error; err != nil {
		return xerrors.Errorf("unable to create new notifications: %w", err)
	}
	n.logNewNotification(&notification)
	return nil
}

func (n *Notifier) onBookingAdded(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.BookingAdded)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	n.logger.Info("OnBookingAdded", log.UInt64("BookingID", payloadCast.BookingId))
	var booking model.Booking
	if err := tx.Preload("RoomStays.Room").
		Preload("Notifications").
		Preload("Guest").
		Preload("Hotel").
		Find(&booking, payloadCast.BookingId).Error; err != nil {
		return xerrors.Errorf("unable to get booking to notify: %w", err)
	}
	var hasConfirmed bool
	var bookingStatuses []string
	var stayStatuses []string
	for _, rs := range booking.RoomStays {
		if rs.BookingStatus == model.BookingStatusConfirmed && rs.Status == model.RoomStayStatusNew {
			hasConfirmed = true
		}
		bookingStatuses = append(bookingStatuses, string(rs.BookingStatus))
		stayStatuses = append(stayStatuses, string(rs.Status))
	}
	if hasConfirmed {
		return n.createNotificationOfType(tx, &booking, newBookingNotification)
	} else {
		n.logger.Info("Non-confirmed booking added, skipping notification",
			log.UInt("BookingID", booking.ID),
			log.String("BookingNumber", booking.TravellineNumber),
			log.String("BookingStatuses", strings.Join(bookingStatuses, ", ")),
			log.String("StayStatuses", strings.Join(stayStatuses, ", ")))
		return nil
	}
}

func (n *Notifier) onBookingConfirmed(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.BookingConfirmed)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	n.logger.Info("OnBookingConfirmed", log.UInt64("BookingID", payloadCast.BookingId))
	var booking model.Booking
	if err := tx.Preload("RoomStays.Room").
		Preload("Notifications").
		Preload("Guest").
		Preload("Hotel").
		Find(&booking, payloadCast.BookingId).Error; err != nil {
		return xerrors.Errorf("unable to get booking to notify: %w", err)
	}
	var hasConfirmed bool
	var bookingStatuses []string
	var stayStatuses []string
	for _, rs := range booking.RoomStays {
		if rs.BookingStatus == model.BookingStatusConfirmed && rs.Status == model.RoomStayStatusNew {
			hasConfirmed = true
		}
		bookingStatuses = append(bookingStatuses, string(rs.BookingStatus))
		stayStatuses = append(stayStatuses, string(rs.Status))
	}
	if hasConfirmed {
		return n.createNotificationOfType(tx, &booking, newBookingNotification)
	} else {
		n.logger.Warn("Non-confirmed booking added, skipping notification",
			log.UInt("BookingID", booking.ID),
			log.String("BookingNumber", booking.TravellineNumber),
			log.String("BookingStatuses", strings.Join(bookingStatuses, ", ")),
			log.String("StayStatuses", strings.Join(stayStatuses, ", ")))
		return nil
	}
}

func (n *Notifier) onRoomStayCheckedIn(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.RoomStayCheckedIn)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	n.logger.Info("OnRoomStayCheckedIn",
		log.UInt64("BookingID", payloadCast.BookingId),
		log.UInt64("RoomStayID", payloadCast.RoomStayId),
	)
	var booking model.Booking
	if err := tx.Preload("RoomStays.Room").
		Preload("Notifications").
		Preload("Guest").
		Preload("Hotel").
		Find(&booking, payloadCast.BookingId).Error; err != nil {
		return xerrors.Errorf("unable to get booking to notify: %w", err)
	}

	return n.createNotificationOfType(tx, &booking, checkedInNotification)
}

func (n *Notifier) onBookingCancelled(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.BookingCancelled)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	n.logger.Info("OnBookingCancelled", log.UInt64("BookingID", payloadCast.BookingId))
	var booking model.Booking
	if err := tx.Preload("RoomStays.Room").
		Preload("Notifications").
		Preload("Guest").
		Preload("Hotel").
		Find(&booking, payloadCast.BookingId).Error; err != nil {
		return xerrors.Errorf("unable to get booking to notify: %w", err)
	}
	return n.createNotificationOfType(tx, &booking, bookingCancelledNotification)
}

func getNotificationLogFields(notification *model.Notification) []log.Field {
	return []log.Field{
		log.UInt("NotificationID", notification.ID),
		log.UInt("BookingID", notification.Booking.ID),
		log.String("BookingNumber", notification.Booking.TravellineNumber),
		log.String("NotificationType", string(notification.Type)),
		log.String("NotificationState", string(notification.State)),
	}
}
