package notifier

import (
	"fmt"
	"strings"
	"time"

	"gorm.io/gorm"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
)

const notificationDateLayout = "02.01.2006"
const notificationTimeLayout = "15:04"

func stayDatesPayloadBuilder(tx *gorm.DB, cfg *Config, notification *model.Notification) error {
	notification.Payload = model.JSONMap{
		"ArrivalDate":   notification.Booking.RoomStays[0].CheckInDateTime.Time.Format(notificationDateLayout),
		"DepartureDate": notification.Booking.RoomStays[0].CheckOutDateTime.Time.Format(notificationDateLayout),
		"CheckInTime":   notification.Booking.RoomStays[0].CheckInDateTime.Time.Format(notificationTimeLayout),
		"CheckOutTime":  notification.Booking.RoomStays[0].CheckOutDateTime.Time.Format(notificationTimeLayout),
	}
	return nil
}

func checkinPayloadBuilder(tx *gorm.DB, cfg *Config, notification *model.Notification) error {
	var roomNumbers []string
	for _, rs := range notification.Booking.RoomStays {
		if rs.BookingStatus != model.BookingStatusConfirmed || rs.Status != model.RoomStayStatusCheckedIn {
			continue
		}
		roomNumbers = append(roomNumbers, rs.Room.Name)
	}

	var roomString string
	if len(roomNumbers) == 0 {
		return xerrors.Errorf("No rooms in booking")
	}
	if len(roomNumbers) == 1 {
		roomString = fmt.Sprintf("номера %s", roomNumbers[0])
	} else {
		allButLast := roomNumbers[0 : len(roomNumbers)-1]
		roomString = fmt.Sprintf("номеров %s и %s", strings.Join(allButLast, ", "), roomNumbers[len(roomNumbers)-1])
	}

	retryPromoType := model.PromoType(fmt.Sprintf("hotel_%s_retry", notification.Booking.Hotel.Key))
	typesToCreate := append(model.GenericPromoTypes, retryPromoType)
	promoPayload := make(map[string]interface{}, 2*len(typesToCreate))
	expirationDate := time.Now().Add(cfg.MinPromoValidity)
	var payloadsToBind []*model.Promo
	for _, t := range typesToCreate {
		p, err := getPromoOfType(tx, t, expirationDate)
		if err != nil {
			return xerrors.Errorf("unable to find available promo of type %s: %w", t, err)
		}
		var typeName string
		if t == retryPromoType {
			typeName = "hotel_retry"
		} else {
			typeName = strings.ToLower(string(t))
		}
		if p != nil {
			promoPayload["promo_"+typeName] = p.Code
			promoPayload["promo_"+typeName+"_expires"] = p.ExpiresAt.Format(notificationDateLayout)
			payloadsToBind = append(payloadsToBind, p)
		} else {
			promoPayload = nil
			break
		}
	}

	notification.Payload = model.JSONMap{
		"RoomNumber": roomString,
	}
	if promoPayload != nil {
		for k, v := range promoPayload {
			notification.Payload[k] = v
		}
		notification.BoundPromos = payloadsToBind
	}
	return nil
}

func marketPayloadBuilder(tx *gorm.DB, cfg *Config, notification *model.Notification) error {
	expirationDate := time.Now().Add(cfg.MinPromoValidity)
	marketPromo, err := getPromoOfType(tx, model.MarketPromoType, expirationDate)
	if err != nil {
		return xerrors.Errorf("unable to find market promo for payload: %w", err)
	}
	if marketPromo != nil {
		notification.Payload = model.JSONMap{
			"promo_market": marketPromo.Code,
		}
		notification.BoundPromos = []*model.Promo{marketPromo}
	}
	return nil
}

func cancellationPayloadBuilder(tx *gorm.DB, cfg *Config, notification *model.Notification) error {
	cancelPromoType := model.PromoType(fmt.Sprintf("hotel_%s_cancel", notification.Booking.Hotel.Key))
	expirationDate := time.Now().Add(cfg.MinPromoValidity)
	cancelPromo, err := getPromoOfType(tx, cancelPromoType, expirationDate)
	if err != nil {
		return xerrors.Errorf("unable to find cancellation promo for payload: %w", err)
	}
	if cancelPromo != nil {
		notification.Payload = model.JSONMap{
			"promo_hotel_cancel":         cancelPromo.Code,
			"promo_hotel_cancel_expires": cancelPromo.ExpiresAt.Format(notificationDateLayout),
		}
		notification.BoundPromos = []*model.Promo{cancelPromo}
	}
	return nil
}

func getPromoOfType(tx *gorm.DB, promoType model.PromoType, expirationDate time.Time) (*model.Promo, error) {
	var promo model.Promo
	res := tx.Where(model.Promo{Type: promoType}).
		Where("notification_id IS NULL").
		Where("expires_at > ?", expirationDate).
		Order("expires_at asc").
		Find(&promo)
	if err := res.Error; err != nil {
		return nil, err
	}
	if res.RowsAffected == 0 {
		return nil, nil
	}
	return &promo, nil
}
