package synchronizer

import (
	"database/sql"
	"time"

	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
	"a.yandex-team.ru/travel/budapest/metapms/internal/webpms"
)

func mapGuestDTO(dto *webpms.GuestDTO) *model.Guest {
	return &model.Guest{
		TravellineID:    dto.ID,
		LastName:        dto.LastName,
		FirstName:       dto.FirstName,
		MiddleName:      dto.MiddleName,
		BirthDate:       mapDate(dto.BirthDate),
		CitizenshipCode: dto.CitizenshipCode,
		Status:          mapDictionaryItemDTO(dto.Status),
		Emails:          dto.Emails,
		Phones:          dto.Phones,
		Gender:          dto.Gender,
	}
}

func mapBookingDTO(dto *webpms.BookingDTO, hotel *model.Hotel, guest *model.Guest) *model.Booking {
	return &model.Booking{
		HotelID:           hotel.ID,
		GuestID:           guest.ID,
		TravellineID:      dto.ID,
		TravellineNumber:  dto.Number,
		CustomerLanguage:  dto.CustomerLanguage,
		VisitPurpose:      mapDictionaryItemDTO(dto.VisitPurpose),
		CustomerComment:   dto.CustomerComment,
		LastModifiedAt:    mapTimestamp(dto.LastModified),
		GroupName:         dto.GroupName,
		CurrencyID:        dto.CurrencyID,
		Source:            mapDictionaryItemDTO(dto.Source),
		SourceChannelName: dto.SourceChannelName,
	}
}

func mapRoomStayDTO(dto *webpms.RoomStayDTO, booking *model.Booking, hotel *model.Hotel, room *model.Room) *model.RoomStay {
	entity := model.RoomStay{
		TravellineID:           dto.ID,
		BookingID:              booking.ID,
		RoomTLID:               dto.RoomID,
		RoomTypeTLID:           dto.RoomTypeID,
		CheckInDateTime:        mapHotelTime(dto.CheckInDateTime, hotel),
		CheckOutDateTime:       mapHotelTime(dto.CheckOutDateTime, hotel),
		ActualCheckInDateTime:  mapHotelTime(dto.ActualCheckInDateTime, hotel),
		ActualCheckOutDateTime: mapHotelTime(dto.ActualCheckOutDateTime, hotel),
		Status:                 model.RoomStayStatus(dto.Status),
		BookingStatus:          model.BookingStatus(dto.BookingStatus),
	}
	if room != nil {
		entity.RoomID = &room.ID
	}
	if dto.GuestCountInfo != nil {
		entity.GuestCountInfo = &model.GuestCountInfo{
			Adults:   dto.GuestCountInfo.Adults,
			Children: dto.GuestCountInfo.Children,
		}
	}
	if dto.Deposit != nil {
		entity.Deposit = &model.Amount{
			DepositAmount:   dto.Deposit.Amount,
			DepositCurrency: dto.Deposit.Currency,
		}
	}
	if dto.TotalPrice != nil {
		entity.TotalPrice = &model.TotalPrice{
			TotalAmount:    dto.TotalPrice.Amount,
			ToPayAmount:    dto.TotalPrice.ToPayAmount,
			ToRefundAmount: dto.TotalPrice.ToRefundAmount,
		}
	}
	entity.Amenities = make(model.CommaStringList, len(dto.Amenities))
	for i, a := range dto.Amenities {
		entity.Amenities[i] = a.Name
	}
	return &entity
}

func mapDictionaryItemDTO(dto *webpms.DictionaryItemDTO) model.JSONMap {
	if dto == nil {
		return nil
	}
	return model.JSONMap{
		"key":   dto.Key,
		"value": dto.Value,
	}
}

func mapTimestamp(ts *webpms.Timestamp) sql.NullTime {
	if ts == nil {
		return sql.NullTime{}
	}

	return sql.NullTime{
		Time:  ts.Time.In(time.Local),
		Valid: true,
	}
}

func mapHotelTime(ht *webpms.HotelTime, hotel *model.Hotel) sql.NullTime {
	if ht == nil {
		return sql.NullTime{}
	}
	location, _ := hotel.GetLocation()
	zoned := ht.ZonedTo(location).In(time.Local)
	return sql.NullTime{
		Time:  zoned,
		Valid: true,
	}
}

func mapDate(d *webpms.Date) *model.Date {
	if d == nil {
		return nil
	}
	dd := model.Date(d.Time)
	return &dd
}
