package synchronizer

import (
	"context"
	"sync"
	"time"

	"gorm.io/gorm"
	"gorm.io/gorm/clause"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/events"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
	"a.yandex-team.ru/travel/budapest/metapms/internal/pgclient"
	"a.yandex-team.ru/travel/budapest/metapms/internal/queue"
	"a.yandex-team.ru/travel/budapest/metapms/internal/webpms"
)

const hotelMetricsAttr = "hotelMetrics"
const hotelAttr = "hotel"
const pmsAttr = "pms"
const resultAttr = "sync-res"
const pmsKeyPrefix = "PMS_API_KEY_"

type Synchronizer struct {
	cfg       Config
	pmsConfig webpms.Config
	logger    log.Logger
	pg        *pgclient.PGClient
	registry  metrics.Registry
	metrics   *synchronizerMetrics
}

type changedField struct {
	OldValue interface{}
	NewValue interface{}
}

type changedEntity struct {
	entity  interface{}
	changes map[string]*changedField
}

type syncResult struct {
	newBookings      []uint
	changedBookings  map[uint]*changedEntity
	changedRoomStays map[uint]*changedEntity
}

func New(syncConfig Config, pmsConfig webpms.Config, pg *pgclient.PGClient, logger log.Logger, registry metrics.Registry) *Synchronizer {
	return &Synchronizer{
		cfg:       syncConfig,
		pmsConfig: pmsConfig,
		logger:    logger.WithName("Synchronizer"),
		pg:        pg,
		registry:  registry,
		metrics:   createSynchronizerMetrics(registry.WithPrefix("synchronizer")),
	}
}

func (s *Synchronizer) Run(ctx context.Context) error {
	if !s.cfg.Enabled {
		return nil
	}
	if _, err := s.SyncOnce(ctx); err != nil {
		return xerrors.Errorf("unable to sync initially: %w", err)
	}
	ticker := time.NewTicker(s.cfg.SyncInterval)
	defer ticker.Stop()
	for {
		select {
		case <-ticker.C:
			for {
				if hasMore, err := s.SyncOnce(ctx); err != nil {
					s.logger.Error("Error while running sync", log.Error(err))
					break
				} else {
					if !hasMore {
						break
					}
				}
			}
		case <-ctx.Done():
			return nil
		}
	}
}

func (s *Synchronizer) SyncOnce(ctx context.Context) (hasMore bool, syncErr error) {
	defer func() {
		s.metrics.syncsRunning.Add(-1)
		if syncErr != nil {
			s.metrics.syncsFailed.Inc()
			s.logger.Error("Sync completed with errors", log.Error(syncErr))
		} else {
			s.metrics.syncsSuccessful.Inc()
			s.logger.Debug("Sync completed", log.Bool("HasMore", hasMore), log.Error(syncErr))
		}
	}()
	s.metrics.syncsRunning.Add(1)
	s.logger.Debug("Started synchronization")
	db, err := s.pg.GetPrimary()
	if err != nil {
		return false, xerrors.Errorf("unable to start sync: %w", err)
	}
	var hotels []*model.Hotel
	if err := db.Where(model.Hotel{SyncEnabled: true}).Find(&hotels).Error; err != nil {
		return false, xerrors.Errorf("unable to list hotels to sync: %w", err)
	}

	tasks := make([]*syncTask, len(hotels))
	for i, hotel := range hotels {
		pmsKey, exists := s.cfg.PMSKeys[hotel.Key]
		if !exists {
			return false, xerrors.Errorf("no pms api key for hotel with key: %s", hotel.Key)
		}
		var pmsConfig = s.pmsConfig
		pmsConfig.APIKey = pmsKey
		task := &syncTask{
			pms:   webpms.NewClient(pmsConfig),
			hotel: hotel,
			result: syncResult{
				changedBookings:  map[uint]*changedEntity{},
				changedRoomStays: map[uint]*changedEntity{},
			},
		}
		tasks[i] = task
	}
	wg := sync.WaitGroup{}
	wg.Add(len(tasks))
	for _, task := range tasks {
		go func(task *syncTask) {
			defer func() {
				wg.Done()
			}()
			if _, err := task.hotel.GetLocation(); err != nil {
				task.err = err
				return
			}
			if err := model.WithChangeCallback(db, s).
				WithContext(ctx).
				Set(hotelAttr, task.hotel).
				Set(hotelMetricsAttr, createHotelSyncMetrics(s.registry, task.hotel)).
				Set(pmsAttr, task.pms).
				Set(resultAttr, &task.result).
				Transaction(func(tx *gorm.DB) error {
					more, err := s.syncHotelWithLocking(tx)
					if more {
						task.hasMore = true
					}
					return err
				}); err != nil {
				task.err = err
			}
		}(task)
	}
	wg.Wait()
	var hasErrors bool
	for _, task := range tasks {
		if task.err != nil {
			if pgclient.IsLockError(task.err) {
				continue
			}
			hasErrors = true
			s.logger.Error("Error while running sync", log.String("HotelKey", task.hotel.Key), log.Error(task.err))
		} else {
			if task.hasMore {
				hasMore = true
			}
		}
	}
	if hasErrors {
		return false, xerrors.Errorf("Some hotels failed to sync")
	}
	return hasMore, nil
}

func (s *Synchronizer) syncHotelWithLocking(tx *gorm.DB) (hasMore bool, syncErr error) {
	m := getHotelMetrics(tx)
	started := time.Now()
	m.syncsRunning.Add(1)
	defer func() {
		m.syncsRunning.Add(-1)
		m.syncTiming.RecordDuration(time.Since(started))
		if syncErr != nil {
			m.syncsFailed.Inc()
		} else {
			m.syncsSuccessful.Inc()
		}
	}()
	syncRoot := &model.Sync{}
	hotel := getHotel(tx)
	res := tx.Clauses(clause.Locking{
		Strength: "UPDATE",
	}).Find(syncRoot, model.Sync{
		Key: hotel.Key,
	})
	if res.Error != nil {
		return false, xerrors.Errorf("unable to load sync object: %w", res.Error)
	}
	if res.RowsAffected == 0 {
		syncRoot.Key = hotel.Key
		s.logger.Info("Sync object not found, will create a new one", log.String("HotelKey", hotel.Key))
		insertion := res.Create(syncRoot)
		if insertion.Error != nil {
			return false, xerrors.Errorf("unable to create sync object: %w", res.Error)
		}
	}

	var intervalStart time.Time
	var intervalEnd time.Time
	if syncRoot.SyncedTill == nil {
		intervalStart = s.cfg.InitialSyncDate
		s.logger.Warn("No lastRun in sync object, expecting full sync since start of the epoch",
			log.String("HotelKey", hotel.Key),
			log.Time("EpochStart", s.cfg.InitialSyncDate))
	} else {
		intervalStart = *syncRoot.SyncedTill
	}

	now := time.Now()
	targetEnd := now.Add(2 * time.Minute) // to protect from clock drift between us and TL
	intervalEnd = targetEnd
	hasMore = false
	if intervalEnd.After(intervalStart.Add(s.cfg.MaxSyncWindow)) {
		intervalEnd = intervalStart.Add(s.cfg.MaxSyncWindow)
		s.logger.Warn("Large interval, will shrink",
			log.String("HotelKey", hotel.Key),
			log.Time("IntervalStart", intervalStart),
			log.Time("IntervalEnd", intervalEnd))
		hasMore = true
	}
	if err := s.syncHotel(tx, hotel, intervalStart, intervalEnd); err != nil {
		return false, err
	}
	syncRoot.LastRun = &now
	var syncedTill time.Time
	if intervalEnd.After(now) {
		syncedTill = now
	} else {
		syncedTill = intervalEnd
	}
	syncRoot.SyncedTill = &syncedTill
	tx.Save(syncRoot)
	return hasMore, nil
}

func (s *Synchronizer) syncHotel(tx *gorm.DB, hotel *model.Hotel, intervalStart time.Time, intervalEnd time.Time) error {
	pms := getPMS(tx)
	rooms, err := s.syncRooms(tx, hotel, pms)
	if err != nil {
		return xerrors.Errorf("unable to sync rooms: %w", err)
	}
	s.logger.Debug("Sync bookings",
		log.String("HotelKey", hotel.Key),
		log.String("Stage", "FindBookings"),
		log.Time("PeriodStart", intervalStart),
		log.Time("PeriodEnd", intervalEnd),
	)

	activeBookingsResp, err := pms.FindBookings(tx.Statement.Context, webpms.BookingLookupState(webpms.BookingLookupStateActive), webpms.Modified(intervalStart, intervalEnd))
	if err != nil {
		return xerrors.Errorf("unable to list active bookings")
	}
	cancelledBookingsResp, err := pms.FindBookings(tx.Statement.Context, webpms.BookingLookupState(webpms.BookingLookupStateCancelled), webpms.Modified(intervalStart, intervalEnd))
	if err != nil {
		return xerrors.Errorf("unable to list cancelled bookings")
	}
	s.logger.Debug("Sync bookings",
		log.String("HotelKey", hotel.Key),
		log.String("Stage", "Listing bookings"),
		log.Int("ActiveBookings", len(activeBookingsResp.BookingNumbers)),
		log.Int("CancelledBookings", len(cancelledBookingsResp.BookingNumbers)),
		log.Time("PeriodStart", intervalStart),
		log.Time("PeriodEnd", intervalEnd),
	)
	numbers := make([]string, len(activeBookingsResp.BookingNumbers)+len(cancelledBookingsResp.BookingNumbers))
	copy(numbers, activeBookingsResp.BookingNumbers)
	for i, n := range cancelledBookingsResp.BookingNumbers {
		numbers[len(activeBookingsResp.BookingNumbers)+i] = n
	}
	if len(numbers) == 0 {
		s.logger.Debug("No new data",
			log.String("HotelKey", hotel.Key),
			log.Time("PeriodStart", intervalStart),
			log.Time("PeriodEnd", intervalEnd))
		return nil
	}
	iterator := webpms.IterateBookings(tx.Statement.Context, pms, numbers)
	for {
		dto, err := iterator.Get()
		if err != nil {
			return xerrors.Errorf("unable to fetch booking data from tl: %w", err)
		}
		if dto == nil {
			break
		}

		var guest model.Guest
		if err := tx.Where(&model.Guest{TravellineID: dto.Customer.ID}).
			Assign(mapGuestDTO(dto.Customer)).
			FirstOrCreate(&guest).Error; err != nil {
			return xerrors.Errorf("unable to synchronize guest with tl id %s: %w", dto.Customer.ID, err)
		}
		s.logger.Debug("Synchronized guest",
			log.String("HotelKey", hotel.Key),
			log.String("GuestTLID", guest.TravellineID),
			log.UInt("ID", guest.ID),
		)
		var booking model.Booking
		if err := tx.Where(&model.Booking{TravellineID: dto.ID}).
			Assign(mapBookingDTO(dto, hotel, &guest)).
			FirstOrCreate(&booking).Error; err != nil {
			return xerrors.Errorf("unable to synchronize booking with tl id %s: %w", dto.ID, err)
		}
		s.logger.Debug("Synchronized booking",
			log.String("HotelKey", hotel.Key),
			log.String("BookingTLID", booking.TravellineID),
			log.UInt("ID", booking.ID),
		)
		for _, rsDTO := range dto.RoomStays {
			var room *model.Room
			var exists bool
			if rsDTO.RoomID != "" {
				room, exists = rooms[rsDTO.RoomID]
				if !exists {
					return xerrors.Errorf("unable to synchronize room stay with tl id %s: room with tl id %s is not found", rsDTO.ID, rsDTO.RoomID)
				}
			}
			var roomStay model.RoomStay
			if err := tx.Where(&model.RoomStay{TravellineID: rsDTO.ID}).
				Assign(mapRoomStayDTO(&rsDTO, &booking, hotel, room)).
				FirstOrCreate(&roomStay).Error; err != nil {
				return xerrors.Errorf("unable to synchronize room stay with tl id %s: %w", rsDTO.ID, err)
			}
			s.logger.Debug("Synchronized roomStay",
				log.String("HotelKey", hotel.Key),
				log.String("RoomStayTLID", roomStay.TravellineID),
				log.UInt("ID", roomStay.ID),
			)
		}
	}
	s.logger.Debug("Sync bookings",
		log.String("HotelKey", hotel.Key),
		log.String("Stage", "Done"),
		log.Time("PeriodStart", intervalStart),
		log.Time("PeriodEnd", intervalEnd),
	)
	return nil
}

func (s *Synchronizer) syncRooms(tx *gorm.DB, hotel *model.Hotel, pms *webpms.Client) (map[string]*model.Room, error) {
	resp, err := pms.ListRooms(tx.Statement.Context)
	if err != nil {
		return nil, xerrors.Errorf("unable to sync rooms: %w", err)
	}
	rooms := map[string]*model.Room{}
	for _, roomDTO := range *resp {
		var room model.Room
		if err := tx.Where(model.Room{TravellineID: roomDTO.ID, HotelID: hotel.ID}).Assign(&model.Room{
			HotelID:      hotel.ID,
			TravellineID: roomDTO.ID,
			Name:         roomDTO.Name,
			RoomTypeTLID: roomDTO.RoomTypeID,
		}).FirstOrCreate(&room).Error; err != nil {
			return nil, xerrors.Errorf("unable to sync room with tl id %s: %w", roomDTO.ID, err)
		}
		rooms[roomDTO.ID] = &room
	}
	return rooms, nil
}

func (s *Synchronizer) OnNewEntity(tx *gorm.DB, entity interface{}) error {
	switch ent := entity.(type) {
	case *model.Booking:
		booking := ent
		if err := s.onNewBooking(tx, booking); err != nil {
			return err
		}
	case *model.RoomStay:
		roomStay := ent
		if err := s.onNewRoomStay(tx, roomStay); err != nil {
			return err
		}
	}
	return nil
}

func (s *Synchronizer) OnFieldsChanged(tx *gorm.DB, entity interface{}, fields []string) error {
	modelValue := tx.Statement.ReflectValue
	changedEnt := changedEntity{
		changes: map[string]*changedField{},
	}
	for _, f := range fields {
		if field := tx.Statement.Schema.LookUpField(f); field != nil {
			fieldValue, _ := field.ValueOf(modelValue)
			changedEnt.changes[f] = &changedField{OldValue: fieldValue}
		}
	}
	if len(changedEnt.changes) > 0 {
		sr := getSyncResult(tx)
		switch ent := entity.(type) {
		case *model.Booking:
			booking := ent
			sr.changedBookings[booking.ID] = &changedEnt
		case *model.RoomStay:
			roomStay := ent
			sr.changedRoomStays[roomStay.ID] = &changedEnt
		}
	}
	return nil
}

func (s *Synchronizer) OnSavedEntity(tx *gorm.DB, entity interface{}) error {
	sr := getSyncResult(tx)
	switch ent := entity.(type) {
	case *model.Booking:
		booking := ent
		change, ok := sr.changedBookings[booking.ID]
		if ok {
			modelValue := tx.Statement.ReflectValue
			for f, v := range change.changes {
				if field := tx.Statement.Schema.LookUpField(f); field != nil {
					newValue, _ := field.ValueOf(modelValue)
					v.NewValue = newValue
				}
			}
			if err := s.onUpdatedBooking(tx, booking, change); err != nil {
				return xerrors.Errorf("unable to handle updated booking: %w", err)
			}
		}

	case *model.RoomStay:
		roomStay := ent
		change, ok := sr.changedRoomStays[roomStay.ID]
		if ok {
			modelValue := tx.Statement.ReflectValue
			for f, v := range change.changes {
				if field := tx.Statement.Schema.LookUpField(f); field != nil {
					newValue, _ := field.ValueOf(modelValue)
					v.NewValue = newValue
				}
			}
			if err := s.onUpdatedRoomStay(tx, roomStay, change); err != nil {
				return xerrors.Errorf("unable to handle updated room stay: %w", err)
			}
		}
	}
	return nil
}

func (s *Synchronizer) onNewRoomStay(tx *gorm.DB, roomStay *model.RoomStay) error {
	getHotelMetrics(tx).roomStaysAdded.Inc()
	return nil
}

func (s *Synchronizer) onNewBooking(tx *gorm.DB, booking *model.Booking) error {
	hotel := getHotel(tx)
	getHotelMetrics(tx).bookingsAdded.Inc()
	s.logger.Info("New booking added", log.String("HotelKey", hotel.Key),
		log.UInt("BookingID", booking.ID),
		log.String("BookingTLID", booking.TravellineID))
	if err := queue.Push(tx, "Bookings", &events.BookingAdded{
		BookingId: uint64(booking.ID),
		HotelKey:  hotel.Key,
	}); err != nil {
		return xerrors.Errorf("unable to notify on new booking: %w", err)
	}
	sr := getSyncResult(tx)
	sr.newBookings = append(sr.newBookings, booking.ID)
	return nil
}

func (s *Synchronizer) onUpdatedBooking(tx *gorm.DB, booking *model.Booking, change *changedEntity) error {
	hotel := getHotel(tx)
	getHotelMetrics(tx).bookingsUpdated.Inc()
	s.logger.Info("Booking updated", log.String("HotelKey", hotel.Key),
		log.String("BookingTLID", booking.TravellineID),
		log.String("BookingNumber", booking.TravellineNumber),
		log.UInt("ID", booking.ID),
		log.Any("ChangedFields", change.changes),
	)
	return nil
}

func (s *Synchronizer) onUpdatedRoomStay(tx *gorm.DB, roomStay *model.RoomStay, change *changedEntity) error {
	hotel := getHotel(tx)
	getHotelMetrics(tx).roomStaysUpdated.Inc()
	if v, changed := change.changes["BookingStatus"]; changed {
		switch {
		case v.NewValue == model.BookingStatusCancelled && v.OldValue == model.BookingStatusConfirmed:
			if err := s.onCancelledBookingRoomStay(tx, roomStay); err != nil {
				return xerrors.Errorf("unable to handle cancelled booking: %w", err)
			}
		case v.NewValue == model.BookingStatusConfirmed && (v.OldValue == model.BookingStatusPending || v.OldValue == model.BookingStatusUnconfirmed):
			if err := queue.Push(tx, "Bookings", &events.BookingConfirmed{
				BookingId: uint64(roomStay.BookingID),
				HotelKey:  hotel.Key,
			}); err != nil {
				return xerrors.Errorf("unable to notify on new booking: %w", err)
			}
		}
	}

	if v, statusChanged := change.changes["Status"]; statusChanged {
		switch v.NewValue {
		case model.RoomStayStatusCheckedIn:
			if err := s.onCheckedInRoomStay(tx, roomStay); err != nil {
				return xerrors.Errorf("unable to handle checkedIn roomStay: %w", err)
			}
		case model.RoomStayStatusCheckedOut:
			if err := s.onCheckedOutRoomStay(tx, roomStay); err != nil {
				return xerrors.Errorf("unable to handle checkedOut roomStay: %w", err)
			}
		case model.RoomStayStatusNew:
			if v.OldValue == model.RoomStayStatusCheckedIn {
				if err := s.onCheckInCancelledRoomStay(tx, roomStay); err != nil {
					return xerrors.Errorf("unable to handle check in cancelled roomStay: %w", err)
				}
			}
		}
	}

	if v, roomChanged := change.changes["RoomID"]; roomChanged {
		if roomStay.Status == model.RoomStayStatusCheckedIn {
			oldRoom, cast1 := v.OldValue.(*uint)
			newRoom, cast2 := v.NewValue.(*uint)
			if cast1 && cast2 && oldRoom != nil && newRoom != nil {
				if err := s.onActiveRoomChanged(tx, roomStay, *oldRoom, *newRoom); err != nil {
					return xerrors.Errorf("unable to handle room changed roomStay: %w", err)
				}
			}
		}
	}

	s.logger.Info("RoomStay updated", log.String("HotelKey", hotel.Key),
		log.UInt("BookingID", roomStay.BookingID),
		log.UInt("ID", roomStay.ID),
		log.Any("ChangedFields", change.changes),
	)
	return nil
}

func (s *Synchronizer) onCancelledBookingRoomStay(tx *gorm.DB, roomStay *model.RoomStay) error {
	hotel := getHotel(tx)
	if err := queue.Push(tx, "Bookings", &events.BookingCancelled{
		BookingId: uint64(roomStay.BookingID),
		HotelKey:  hotel.Key,
	}); err != nil {
		return xerrors.Errorf("unable to notify on cancelled booking: %w", err)
	}
	s.logger.Info("Booking cancelled", log.String("HotelKey", hotel.Key),
		log.UInt("BookingID", roomStay.BookingID))
	return nil
}

func (s *Synchronizer) onCheckedInRoomStay(tx *gorm.DB, roomStay *model.RoomStay) error {
	hotel := getHotel(tx)
	if err := queue.Push(tx, "Bookings", &events.RoomStayCheckedIn{
		BookingId:  uint64(roomStay.BookingID),
		RoomStayId: uint64(roomStay.ID),
		HotelKey:   hotel.Key,
	}); err != nil {
		return xerrors.Errorf("unable to notify on checkedIn room stay: %w", err)
	}
	s.logger.Info("RoomStay checkedIn", log.String("HotelKey", hotel.Key),
		log.UInt("BookingID", roomStay.BookingID),
		log.UInt("RoomStay", roomStay.ID))
	return nil
}

func (s *Synchronizer) onCheckedOutRoomStay(tx *gorm.DB, roomStay *model.RoomStay) error {
	hotel := getHotel(tx)
	if err := queue.Push(tx, "Bookings", &events.RoomStayCheckedOut{
		BookingId:  uint64(roomStay.BookingID),
		RoomStayId: uint64(roomStay.ID),
		HotelKey:   hotel.Key,
	}); err != nil {
		return xerrors.Errorf("unable to notify on checkedOut room stay: %w", err)
	}
	s.logger.Info("RoomStay checkedOut", log.String("HotelKey", hotel.Key),
		log.UInt("BookingID", roomStay.BookingID),
		log.UInt("RoomStay", roomStay.ID))
	return nil
}

func (s *Synchronizer) onActiveRoomChanged(tx *gorm.DB, roomStay *model.RoomStay, oldRoomID uint, newRoomID uint) error {
	hotel := getHotel(tx)
	if err := queue.Push(tx, "Bookings", &events.RoomStayChangedRoom{
		BookingId:  uint64(roomStay.BookingID),
		RoomStayId: uint64(roomStay.ID),
		HotelKey:   hotel.Key,
		OldRoomId:  uint64(oldRoomID),
		NewRoomId:  uint64(newRoomID),
	}); err != nil {
		return xerrors.Errorf("unable to notify on changeRoom room stay: %w", err)
	}
	s.logger.Info("RoomStay room changed", log.String("HotelKey", hotel.Key),
		log.UInt("BookingID", roomStay.BookingID),
		log.UInt("RoomStay", roomStay.ID),
		log.UInt("OldRoomID", oldRoomID),
		log.UInt("NewRoomID", newRoomID))
	return nil
}

func (s *Synchronizer) onCheckInCancelledRoomStay(tx *gorm.DB, roomStay *model.RoomStay) error {
	hotel := getHotel(tx)
	if err := queue.Push(tx, "Bookings", &events.RoomStayCheckinCancelled{
		BookingId:  uint64(roomStay.BookingID),
		RoomStayId: uint64(roomStay.ID),
		HotelKey:   hotel.Key,
	}); err != nil {
		return xerrors.Errorf("unable to notify on checkin cancelled room stay: %w", err)
	}
	s.logger.Info("RoomStay checkIn cancelled", log.String("HotelKey", hotel.Key),
		log.UInt("BookingID", roomStay.BookingID),
		log.UInt("RoomStay", roomStay.ID))
	return nil
}

func getHotel(tx *gorm.DB) *model.Hotel {
	h, found := tx.Get(hotelAttr)
	if !found {
		panic("no hotel in tx")
	}
	return h.(*model.Hotel)
}

func getPMS(tx *gorm.DB) *webpms.Client {
	pmsInt, found := tx.Get(pmsAttr)
	if !found {
		panic("no pms in tx")
	}
	pms := pmsInt.(*webpms.Client)
	return pms
}

func getSyncResult(tx *gorm.DB) *syncResult {
	resInt, found := tx.Get(resultAttr)
	if !found {
		panic("no sync result in tx")
	}
	return resInt.(*syncResult)
}

func getHotelMetrics(tx *gorm.DB) *hotelSyncMetrics {
	mInt, found := tx.Get(hotelMetricsAttr)
	if !found {
		panic("no hotel metrics in tx")
	}
	return mInt.(*hotelSyncMetrics)
}

type syncTask struct {
	pms     *webpms.Client
	hotel   *model.Hotel
	hasMore bool
	err     error
	result  syncResult
}
