package tgbot

import (
	"context"
	"sync"
	"time"

	tb "gopkg.in/tucnak/telebot.v2"
	"gorm.io/gorm"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/metrics"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/alice4business"
	"a.yandex-team.ru/travel/budapest/metapms/internal/events"
	"a.yandex-team.ru/travel/budapest/metapms/internal/pgclient"
	"a.yandex-team.ru/travel/budapest/metapms/internal/queue"
)

type TGBot struct {
	tg             *tb.Bot
	a4b            A4B
	pg             *pgclient.PGClient
	logger         log.Logger
	cfg            Config
	activatorQueue *queue.Queue
	bookingQueue   *queue.Queue
	reports        map[uint]*HotelReport
	syncRoot       sync.Mutex
}

var roomButton = tb.InlineButton{
	Unique: "room_details",
}

var roomListButton = tb.InlineButton{
	Unique: "hotel_room_list",
}

var hotelDetailsButton = tb.InlineButton{
	Unique: "hotel_details",
}

var hotelUpdateButton = tb.InlineButton{
	Unique: "hotel_update",
}

func New(cfg Config, pg *pgclient.PGClient, logger log.Logger, registry metrics.Registry) (*TGBot, error) {
	logger = logger.WithName("TGBot")
	settings := tb.Settings{
		Token:  cfg.TG.Token,
		Poller: &tb.LongPoller{Timeout: cfg.TG.Timeout},
	}
	tg, err := tb.NewBot(settings)
	if err != nil {
		return nil, xerrors.Errorf("unable to create tg-bot: %w", err)
	}

	b := TGBot{
		cfg:            cfg,
		tg:             tg,
		pg:             pg,
		a4b:            alice4business.NewClient(cfg.A4B),
		logger:         logger,
		activatorQueue: queue.New("Activator", "TG-Bot-Activator", pg, logger, registry),
		bookingQueue:   queue.New("Bookings", "TG-Bot-Bookings", pg, logger, registry),
		reports:        make(map[uint]*HotelReport),
		syncRoot:       sync.Mutex{},
	}
	b.addHandler("/subscribe", true, b.handleSubscribeCommand)
	b.addHandler("/unsubscribe", true, b.handleUnsubscribeCommand)
	b.addHandler("/status", false, b.handleStatusCommand)
	b.addHandler("/toggledetails", true, b.handleToggleDetailCommand)
	b.addHandler("/rooms", false, b.handleRoomsCommand)
	tg.Handle(&roomButton, b.handleRoomButton)
	tg.Handle(&roomListButton, b.handleRoomListButton)
	tg.Handle(&hotelDetailsButton, b.handleHotelButton)
	tg.Handle(&hotelUpdateButton, b.handleHotelUpdateButton)
	b.bookingQueue.
		Subscribe(&events.RoomStayCheckedIn{}, b.handleRoomStayCheckedInEvent).
		Subscribe(&events.RoomStayCheckedOut{}, b.handleRoomStayCheckedOutEvent).
		Subscribe(&events.RoomStayChangedRoom{}, b.handleRoomStayChangedRoomEvent).
		Subscribe(&events.RoomStayCheckinCancelled{}, b.handleRoomStayCheckinCancelledEvent)
	b.activatorQueue.
		Subscribe(&events.StatusReportRequested{}, b.handleStatusReportRequestedEvent).
		Subscribe(&events.OperationFailed{}, b.handleOperationFailedEvent).
		Subscribe(&events.OperationCancelled{}, b.handleOperationCancelledEvent)

	return &b, nil
}

func (b *TGBot) Run() {
	for {
		err := b.runWithLock(context.Background())
		if err != nil {
			if pgclient.IsLockError(err) {
				b.logger.Debug("Lock conflict: another instance of bot is probably running")
				time.Sleep(b.cfg.LockConflictRetryInterval)
			} else {
				b.logger.Error("Unexpected error while running tg-bot", log.Error(err))
				time.Sleep(b.cfg.ErrorRetryInterval)
			}
		} else {
			b.logger.Warn("TG-Bot stopped, going to restart")
			time.Sleep(b.cfg.ErrorRetryInterval)
			continue
		}
	}
}

func (b *TGBot) runWithLock(ctx context.Context) error {
	return pgclient.WithLock(ctx, b.pg, b.logger, b.cfg.BotLockID, func(lockCtx context.Context, tx *gorm.DB) error {
		go func() {
			<-lockCtx.Done()
			b.logger.Info("Stopping telegram bot")
			b.tg.Stop()
		}()
		b.logger.Info("Starting telegram bot to listen")
		b.start(lockCtx)
		b.logger.Info("Stopped telegram bot")
		return nil
	})
}

func (b *TGBot) start(ctx context.Context) {
	wg := sync.WaitGroup{}
	wg.Add(3)
	go func() {
		defer wg.Done()
		b.tg.Start()
	}()
	go func() {
		defer wg.Done()
		b.logger.Info("Starting listening for booking events")
		err := b.bookingQueue.Listen(ctx, b.cfg.QueuePollInterval)
		if err != nil {
			b.logger.Error("Error while listening for booking events", log.Error(err))
		} else {
			b.logger.Info("Stopped listening for booking events")
		}
	}()
	go func() {
		defer wg.Done()
		b.logger.Info("Starting listening for activator events")
		err := b.activatorQueue.Listen(ctx, b.cfg.QueuePollInterval)
		if err != nil {
			b.logger.Error("Error while listening for activator event", log.Error(err))
		} else {
			b.logger.Info("Stopped listening for activator events")
		}
	}()
	wg.Wait()
}

func (b *TGBot) addHandler(command string, requiresAdmin bool, f func(tx *gorm.DB, message *tb.Message) interface{}) {
	name := command[1:]
	b.tg.Handle(command, func(message *tb.Message) {
		b.handle(name, requiresAdmin, message, f)
	})
}

func (b *TGBot) handle(name string, requiresAdmin bool, message *tb.Message, handleFunc func(tx *gorm.DB, message *tb.Message) interface{}) {
	b.logger.Info("Handling command",
		log.String("Command", name),
		log.String("Payload", message.Payload),
		log.Int("SenderID", message.Sender.ID),
		log.String("SenderName", message.Sender.Username),
		log.Int64("ChatID", message.Chat.ID),
		log.String("ChatType", string(message.Chat.Type)),
		log.String("ChatName", message.Chat.Title),
	)
	db, err := b.pg.GetPrimary()
	if err != nil {
		b.logger.Error("Unable to get DB connection to handle command",
			log.Error(err))
		b.replyOnError(message, err)
		return
	}
	if requiresAdmin {
		if err := b.checkAdmin(db, message); err != nil {
			b.replyOnError(message, err)
			return
		}
	}
	if result := handleFunc(db, message); result != nil {
		switch t := result.(type) {
		case error:
			b.logger.Error("Error while handling command",
				log.String("Command", name),
				log.String("Payload", message.Payload),
				log.Int("SenderID", message.Sender.ID),
				log.String("SenderName", message.Sender.Username),
				log.Int64("ChatID", message.Chat.ID),
				log.String("ChatType", string(message.Chat.Type)),
				log.String("ChatName", message.Chat.Title),
				log.Error(t),
			)
			b.replyOnError(message, t)
		case string:
			b.replyWithText(message, t)
		case Message:
			var options []interface{}

			if t.isHTML {
				options = append(options, &tb.SendOptions{ParseMode: tb.ModeHTML, DisableWebPagePreview: true})
			}
			if t.buttons != nil {
				options = append(options, &tb.ReplyMarkup{InlineKeyboard: t.buttons})
			}
			b.replyWithText(message, t.text, options...)
		}
	}
}
