package tgbot

import (
	"fmt"
	"strings"
	"time"

	tb "gopkg.in/tucnak/telebot.v2"
	"gorm.io/gorm"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
)

func (b *TGBot) getSubscription(tx *gorm.DB, message *tb.Message) (*model.TelegramSubscription, interface{}) {
	subs, err := getChannelSubscriptions(tx, message.Chat.ID)
	if err != nil {
		return nil, err
	}
	if len(subs) == 0 {
		return nil, "Нет активных подписок"
	}
	var s *model.TelegramSubscription
	if len(subs) > 1 {
		if message.Payload == "" {
			keys := make([]string, len(subs))
			i := 0
			for k := range subs {
				keys[i] = k
				i++
			}
			return nil, fmt.Sprintf("В чате более одного отеля, "+
				"укажите после имени команды ключ того, который имеете в виде (на выбор: %s)", strings.Join(keys, ", "))
		} else {
			var found bool
			s, found = subs[message.Payload]
			if !found {
				keys := make([]string, len(subs))
				i := 0
				for k := range subs {
					keys[i] = k
					i++
				}
				return nil, fmt.Sprintf("Отеля с ключом %s нет в чате, "+
					"укажите после имени команды ключ того, который имеете в виде (на выбор: %s)", message.Payload,
					strings.Join(keys, ", "))
			}
		}
	} else {
		for k := range subs {
			s = subs[k]
			break
		}
	}
	return s, nil
}

func addSubscription(tx *gorm.DB, hotel *model.Hotel, message *tb.Message) error {
	var sub model.TelegramSubscription
	if err := tx.Where(model.TelegramSubscription{
		ChatID:  message.Chat.ID,
		HotelID: hotel.ID,
	}).Assign(model.TelegramSubscription{
		Enabled:    true,
		Subscriber: message.Sender.Username,
	}).FirstOrCreate(&sub).Error; err != nil {
		return xerrors.Errorf("unable to create subscription: %w", err)
	}
	return nil
}

func getChannelSubscriptions(tx *gorm.DB, chatID int64) (map[string]*model.TelegramSubscription, error) {
	var subscriptions []*model.TelegramSubscription
	if err := tx.Where(model.TelegramSubscription{ChatID: chatID, Enabled: true}).
		Preload("Hotel").
		Preload("Hotel.Sync").
		Find(&subscriptions).Error; err != nil {
		return nil, err
	}
	result := make(map[string]*model.TelegramSubscription, len(subscriptions))
	for _, s := range subscriptions {
		result[s.Hotel.Key] = s
	}
	return result, nil
}

func getHotelByID(tx *gorm.DB, id uint) (*model.Hotel, error) {
	var hotel model.Hotel
	if err := tx.Model(model.Hotel{}).Joins("Sync").First(&hotel, id).Error; err != nil {
		if xerrors.Is(err, gorm.ErrRecordNotFound) {
			return nil, newTgError("Неизвестный отель")
		} else {
			return nil, wrapTgError("Ошибка поиска отеля", err)
		}
	}
	return &hotel, nil
}

func getHotelByKey(tx *gorm.DB, hotelKey string) (*model.Hotel, error) {
	var hotel model.Hotel
	if err := tx.Where(model.Hotel{
		Key: hotelKey,
	}).Joins("Sync").First(&hotel).Error; err != nil {
		if xerrors.Is(err, gorm.ErrRecordNotFound) {
			return nil, newTgError("Неизвестный отель")
		} else {
			return nil, wrapTgError("Ошибка поиска отеля", err)
		}
	}
	return &hotel, nil
}

func getRoomByID(tx *gorm.DB, roomID uint64) (*model.Room, error) {
	var room model.Room
	if err := tx.Model(model.Room{}).
		Preload("Hotel").
		First(&room, roomID).
		Error; err != nil {
		return nil, xerrors.Errorf("unable to get room by id: %w", err)
	}
	return &room, nil
}

func getRoomStayByID(tx *gorm.DB, roomStayID uint64) (*model.RoomStay, error) {
	var roomStay model.RoomStay
	if err := tx.Model(model.RoomStay{}).
		Preload("Room").
		Preload("Room.Hotel").
		First(&roomStay, roomStayID).
		Error; err != nil {
		return nil, xerrors.Errorf("unable to get roomstay by id: %w", err)
	}
	return &roomStay, nil
}

func getOperationByID(tx *gorm.DB, operationID uint64) (*model.Operation, error) {
	var operation model.Operation
	if err := tx.Model(model.Operation{}).
		Preload("RoomStay").
		Preload("RoomStay.Room").
		Preload("RoomStay.Room.Hotel").
		First(&operation, operationID).
		Error; err != nil {
		return nil, xerrors.Errorf("unable to get roomstay by id: %w", err)
	}
	return &operation, nil
}

func getHotels(tx *gorm.DB, key string) ([]*model.Hotel, error) {
	var hotels []*model.Hotel
	q := tx.
		Model(&model.Hotel{}).
		Joins("Sync").
		Where(model.Hotel{SyncEnabled: true})
	if key != "" {
		q.Where(model.Hotel{Key: key})
	}
	if err := q.Find(&hotels).Error; err != nil {
		return nil, xerrors.Errorf("unable to list hotels: %w", err)
	}
	return hotels, nil
}

func getHotelSubscriptions(tx *gorm.DB, hotel *model.Hotel, detailedOnly bool) ([]*model.TelegramSubscription, error) {
	var subscriptions []*model.TelegramSubscription
	where := model.TelegramSubscription{HotelID: hotel.ID, Enabled: true}
	if detailedOnly {
		where.Detailed = true
	}
	if err := tx.Model(model.TelegramSubscription{}).
		Where(where).
		Preload("Hotel").
		Preload("Hotel.Sync").
		Find(&subscriptions).Error; err != nil {
		return nil, err
	}
	return subscriptions, nil
}

func getRunningOperations(tx *gorm.DB, hotel *model.Hotel) (map[uint]*model.Operation, error) {
	var opSlice []*model.Operation
	subquery := tx.
		Model(model.Operation{}).
		Select("min(id)").
		Where("State in ?", []model.OperationState{model.OperationStateNew, model.OperationStateRunning, model.OperationStateCancelling}).
		Group("room_binding_id")
	query := tx.
		Model(model.Operation{}).
		Preload("RoomBinding").
		Joins("LEFT JOIN room_bindings RoomBinding ON operations.room_binding_id = RoomBinding.id LEFT JOIN rooms Room ON RoomBinding.room_id = Room.id").
		Where("operations.ID in (?)", subquery).
		Where("hotel_id", hotel.ID)
	if err := query.
		Find(&opSlice).Error; err != nil {
		return nil, xerrors.Errorf("unable to load running operations: %w", err)
	}
	opMap := make(map[uint]*model.Operation, len(opSlice))
	for _, op := range opSlice {
		opMap[op.RoomBinding.RoomID] = op
	}
	return opMap, nil
}

func getActiveStays(tx *gorm.DB, hotel *model.Hotel) (map[uint]*model.RoomStay, error) {
	now := time.Now()
	dayStart := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, now.Location())
	var stays []*model.RoomStay
	sql := `SELECT distinct on (room_id) rs.id
			  	FROM room_stays rs
 				JOIN rooms r ON rs.room_id = r.id
				WHERE rs.status = ? AND check_out_date_time > ? AND r.hotel_id = ?
				ORDER BY room_id, check_in_date_time DESC`
	roomStaysQuery := tx.Raw(sql, model.RoomStayStatusCheckedIn, dayStart, hotel.ID)

	if err := tx.Model(model.RoomStay{}).
		Where("room_stays.ID in (?)", roomStaysQuery).
		Joins("Booking").
		Find(&stays).Error; err != nil {
		return nil, xerrors.Errorf("unable to list active stays: %w", err)
	}

	result := make(map[uint]*model.RoomStay, len(stays))
	for _, s := range stays {
		result[*s.RoomID] = s
	}
	return result, nil
}

func getUpcomingStays(tx *gorm.DB, hotel *model.Hotel) (map[uint]*model.RoomStay, error) {
	var stays []*model.RoomStay
	now := time.Now()
	dayStart := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, now.Location())
	sql := `SELECT distinct on (room_id) rs.id
			  	FROM room_stays rs
 				JOIN rooms r ON rs.room_id = r.id
				WHERE status = ? AND check_in_date_time > ? AND r.hotel_id = ?
				ORDER BY room_id, check_in_date_time ASC`
	roomStaysQuery := tx.Raw(sql, model.RoomStayStatusNew, dayStart, hotel.ID)
	if err := tx.Model(model.RoomStay{}).
		Where("room_stays.ID in (?)", roomStaysQuery).
		Joins("Booking").
		Find(&stays).Error; err != nil {
		return nil, xerrors.Errorf("unable to list upcoming stays: %w", err)
	}
	result := make(map[uint]*model.RoomStay, len(stays))
	for _, s := range stays {
		result[*s.RoomID] = s
	}
	return result, nil
}

func getPastStays(tx *gorm.DB, hotel *model.Hotel) (map[uint]*model.RoomStay, error) {
	var stays []*model.RoomStay
	sql := `SELECT distinct on (room_id) rs.id
			  	FROM room_stays rs
 				JOIN rooms r ON rs.room_id = r.id
				WHERE rs.status = ? AND r.hotel_id = ? AND actual_check_out_date_time is not null
				ORDER BY room_id, actual_check_out_date_time DESC`
	roomStaysQuery := tx.Raw(sql, model.RoomStayStatusCheckedOut, hotel.ID)

	if err := tx.Model(model.RoomStay{}).
		Where("room_stays.ID in (?)", roomStaysQuery).
		Joins("Booking").
		Find(&stays).Error; err != nil {
		return nil, xerrors.Errorf("unable to list active stays: %w", err)
	}

	result := make(map[uint]*model.RoomStay, len(stays))
	for _, s := range stays {
		result[*s.RoomID] = s
	}
	return result, nil
}
