package tgbot

import (
	"fmt"
	"strconv"
	"strings"
	"sync"

	"google.golang.org/protobuf/proto"
	tb "gopkg.in/tucnak/telebot.v2"
	"gorm.io/gorm"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/events"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
)

const numRoomColumns = 5

func (b *TGBot) handleSubscribeCommand(tx *gorm.DB, message *tb.Message) interface{} {
	hotelKey := message.Payload
	if hotelKey == "" {
		return newTgError("Не указан идентификатор отеля. Укажите его как `/subscribe <HotelKey>`")
	}
	hotel, err := getHotelByKey(tx, hotelKey)
	if err != nil {
		return err
	}
	if err := addSubscription(tx, hotel, message); err != nil {
		return err
	} else {
		return fmt.Sprintf("Подписка запущена: буду писать сюда статус отеля «%s»", hotel.Name)
	}
}

func (b *TGBot) handleUnsubscribeCommand(tx *gorm.DB, message *tb.Message) interface{} {
	s, err := b.getSubscription(tx, message)
	if err != nil {
		return err
	}
	s.Enabled = false
	if err := tx.Save(&s).Error; err != nil {
		return err
	}
	return fmt.Sprintf("Готово, больше не буду слать сюда уведомления об отеле «%s»", s.Hotel.Name)
}

func (b *TGBot) handleToggleDetailCommand(tx *gorm.DB, message *tb.Message) interface{} {
	s, err := b.getSubscription(tx, message)
	if err != nil {
		return err
	}
	var msg string
	if s.Detailed {
		s.Detailed = false
		msg = "больше не буду"
	} else {
		s.Detailed = true
		msg = "теперь буду"
	}
	if err := tx.Save(&s).Error; err != nil {
		return err
	}
	return fmt.Sprintf("Готово, %s слать сюда уведомления о заездах и выездах в отеле «%s»", msg, s.Hotel.Name)
}

func (b *TGBot) handleStatusCommand(tx *gorm.DB, message *tb.Message) interface{} {
	s, err := b.getSubscription(tx, message)
	if err != nil {
		return err
	}
	report, err := b.getReportForHotel(s.Hotel, true)
	if err != nil {
		return err
	}
	btn := roomListButton
	btn.Data = fmt.Sprintf("%d", s.Hotel.ID)
	btn.Text = "Подробнее"
	return Message{text: report.printRoomsReport(true), isHTML: true, buttons: [][]tb.InlineButton{{btn}}}
}

func (b *TGBot) handleRoomListButton(c *tb.Callback) {
	b.logger.Debug("Button pressed", log.String("ButtonType", "RoomList"), log.String("Data", c.Data))
	defer func() {
		_ = b.tg.Respond(c)
	}()
	hotelID, err := strconv.Atoi(c.Data)
	if err != nil {
		b.logger.Error("Invalid hotel id")
		return
	}
	db, err := b.pg.GetPrimary()
	if err != nil {
		b.logger.Error("Unable to get db to fetch room info", log.Error(err))
		return
	}
	hotel, err := getHotelByID(db, uint(hotelID))
	if err != nil {
		b.logger.Error("Unable to get hotel", log.Error(err))
	}
	title, markup, err := b.buildMessageWithHotelRooms(hotel)
	if err != nil {
		b.logger.Error("Unable to get hotel rooms", log.Error(err))
		return
	}
	_, err = b.tg.Edit(c.Message, title, markup)
	if err != nil {
		b.logger.Error("Unable to edit the message", log.Error(err))
		return
	}
}

func (b *TGBot) handleHotelButtonHelper(c *tb.Callback, force bool) {
	defer func() {
		_ = b.tg.Respond(c)
	}()
	hotelID, err := strconv.Atoi(c.Data)
	if err != nil {
		b.logger.Error("Invalid hotel id")
		return
	}
	db, err := b.pg.GetPrimary()
	if err != nil {
		b.logger.Error("Unable to get db to fetch room info", log.Error(err))
		return
	}
	hotel, err := getHotelByID(db, uint(hotelID))
	if err != nil {
		b.logger.Error("Unable to get hotel", log.Error(err))
	}
	report, err := b.getReportForHotel(hotel, force)
	if err != nil {
		b.logger.Error("Unable to get hotel report", log.Error(err))
		return
	}
	btn1 := hotelUpdateButton
	btn1.Data = fmt.Sprintf("%d", hotel.ID)
	btn1.Text = "Обновить"
	btn2 := roomListButton
	btn2.Data = fmt.Sprintf("%d", hotel.ID)
	btn2.Text = "Подробнее"
	var markup tb.ReplyMarkup
	markup.InlineKeyboard = [][]tb.InlineButton{{btn1}, {btn2}}
	_, err = b.tg.Edit(c.Message, report.printRoomsReport(true), &tb.SendOptions{ParseMode: tb.ModeHTML, DisableWebPagePreview: true}, &markup)
	if err != nil {
		b.logger.Error("Unable to edit the message", log.Error(err))
		return
	}
}

func (b *TGBot) handleHotelUpdateButton(c *tb.Callback) {
	b.logger.Debug("Button pressed", log.String("ButtonType", "HotelUpdate"), log.String("Data", c.Data))
	b.handleHotelButtonHelper(c, true)
}

func (b *TGBot) handleHotelButton(c *tb.Callback) {
	b.logger.Debug("Button pressed", log.String("ButtonType", "Hotel"), log.String("Data", c.Data))
	b.handleHotelButtonHelper(c, false)
}

func (b *TGBot) handleRoomButton(c *tb.Callback) {
	b.logger.Debug("Button pressed", log.String("ButtonType", "Room"), log.String("Data", c.Data))
	defer func() {
		_ = b.tg.Respond(c)
	}()
	parts := strings.Split(c.Data, "-")
	if len(parts) != 2 {
		b.logger.Error("Invalid button data", log.String("Data", c.Data))
		return
	}
	hotelID, err := strconv.Atoi(parts[0])
	if err != nil {
		b.logger.Error("Invalid hotel id")
		return
	}
	roomID, err := strconv.Atoi(parts[1])
	if err != nil {
		b.logger.Error("Invalid room id")
		return
	}
	db, err := b.pg.GetPrimary()
	if err != nil {
		b.logger.Error("Unable to get db to fetch room info", log.Error(err))
		return
	}
	hotel, err := getHotelByID(db, uint(hotelID))
	if err != nil {
		b.logger.Error("Unable to get hotel", log.Error(err))
		return
	}
	roomIDUint := uint(roomID)
	report, err := b.getReportForHotel(hotel, false)
	if err != nil {
		b.logger.Error("Unable to get hotel report", log.Error(err))
		return
	}
	rrd := report.getRoomReport(roomIDUint)
	if rrd == nil {
		b.logger.Error("No room report description")
		return
	}
	description := rrd.getDetailsDescription(true)
	updateButton := roomButton
	updateButton.Data = c.Data
	updateButton.Text = "Обновить"
	backButtonCopy := roomListButton
	backButtonCopy.Text = "« Назад"
	backButtonCopy.Data = fmt.Sprintf("%d", hotelID)
	replyMarkup := tb.ReplyMarkup{InlineKeyboard: [][]tb.InlineButton{
		{
			updateButton,
		},
		{
			backButtonCopy,
		},
	}}
	_, err = b.tg.Edit(c.Message, description, &tb.SendOptions{ParseMode: tb.ModeHTML, DisableWebPagePreview: true}, &replyMarkup)
	if err != nil {
		b.logger.Error("Unable to edit the message", log.Error(err))
		return
	}
}

func (b *TGBot) buildMessageWithHotelRooms(hotel *model.Hotel) (string, *tb.ReplyMarkup, error) {
	report, err := b.getReportForHotel(hotel, false)
	if err != nil {
		return "", nil, err
	}
	var replyMarkup = &tb.ReplyMarkup{}
	replyMarkup.InlineKeyboard = make([][]tb.InlineButton, len(report.roomReportData)/numRoomColumns+1)

	for row := 0; row <= len(report.roomReportData)/numRoomColumns; row++ {
		replyMarkup.InlineKeyboard[row] = make([]tb.InlineButton, numRoomColumns)
		for col := 0; col < numRoomColumns; col++ {
			i := row*numRoomColumns + col
			if i >= len(report.roomReportData) {
				break
			}
			btn := roomButton
			btn.Text = report.roomReportData[i].getNameForRoomList()
			btn.Data = fmt.Sprintf("%d-%d", hotel.ID, report.roomReportData[i].roomID)
			replyMarkup.InlineKeyboard[row][col] = btn
		}
	}
	btn := hotelDetailsButton
	btn.Data = fmt.Sprintf("%d", hotel.ID)
	btn.Text = "« Назад"

	replyMarkup.InlineKeyboard = append(replyMarkup.InlineKeyboard, []tb.InlineButton{
		btn,
	})
	return fmt.Sprintf("Номера отеля %s:", hotel.Name), replyMarkup, nil
}

func (b *TGBot) handleRoomsCommand(tx *gorm.DB, message *tb.Message) interface{} {
	s, err := b.getSubscription(tx, message)
	if err != nil {
		return err
	}
	title, markup, err := b.buildMessageWithHotelRooms(s.Hotel)
	_, sendErr := b.tg.Send(message.Chat, title, markup)
	if sendErr != nil {
		b.logger.Error("Unable to send message", log.Error(sendErr))
	}
	return nil
}

func (b *TGBot) handleRoomStayCheckedInEvent(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.RoomStayCheckedIn)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	b.broadCastRoomStayStatus(tx, payloadCast.RoomStayId, "заселен")
	return nil
}

func (b *TGBot) handleRoomStayCheckedOutEvent(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.RoomStayCheckedOut)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	b.broadCastRoomStayStatus(tx, payloadCast.RoomStayId, "выселен")
	return nil
}

func (b *TGBot) handleRoomStayChangedRoomEvent(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.RoomStayChangedRoom)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	b.broadCastRoomChange(tx, payloadCast.OldRoomId, payloadCast.NewRoomId)
	return nil
}

func (b *TGBot) handleRoomStayCheckinCancelledEvent(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.RoomStayCheckinCancelled)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	b.broadCastRoomStayStatus(tx, payloadCast.RoomStayId, "заселение отменено")
	return nil
}

func (b *TGBot) handleStatusReportRequestedEvent(db *gorm.DB, payload proto.Message) error {
	b.logger.Debug("Status report requested by queue event")
	payloadCast, ok := payload.(*events.StatusReportRequested)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	key := ""
	if !payloadCast.All {
		key = payloadCast.HotelKey
	}
	tx, err := b.pg.GetPrimary()
	if err != nil {
		b.logger.Error("Unable to get db connection to handle status report", log.Error(err))
	}
	hotels, err := getHotels(tx, key)
	if err != nil {
		b.logger.Error("")
	}
	wg := sync.WaitGroup{}
	for _, h := range hotels {
		hotel := h
		subs, err := getHotelSubscriptions(tx, hotel, false)
		if err != nil {
			b.logger.Error("Unable to get hotel subscriptions", log.String("HotelKey", hotel.Key), log.Error(err))
			continue
		}
		if len(subs) == 0 {
			b.logger.Debug("No subscriptions for hotel status", log.String("HotelKey", hotel.Key))
			continue
		}
		wg.Add(1)
		b.logger.Debug("Will generate report for hotel", log.String("HotelKey", hotel.Key))
		go func() {
			defer wg.Done()
			report, err := b.getReportForHotel(hotel, true)
			if err != nil {
				b.logger.Error("Unable to build status report", log.String("HotelKey", hotel.Key), log.Error(err))
				return
			}
			b.logger.Debug("Hotel report generated", log.String("HotelKey", hotel.Key))
			chatIDs := make([]int64, len(subs))
			for i, s := range subs {
				chatIDs[i] = s.ChatID
			}
			btn1 := hotelUpdateButton
			btn1.Data = fmt.Sprintf("%d", hotel.ID)
			btn1.Text = "Обновить"
			btn2 := roomListButton
			btn2.Data = fmt.Sprintf("%d", hotel.ID)
			btn2.Text = "Подробнее"
			var markup tb.ReplyMarkup
			markup.InlineKeyboard = [][]tb.InlineButton{{btn1}, {btn2}}
			b.broadcastText(chatIDs, report.printRoomsReport(true), &tb.SendOptions{ParseMode: tb.ModeHTML, DisableWebPagePreview: true}, &markup)
		}()
	}
	wg.Wait()
	return nil
}

func (b *TGBot) handleOperationCancelledEvent(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.OperationCancelled)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	b.broadCastOperationStatus(tx, payloadCast.OperationId)
	return nil
}

func (b *TGBot) handleOperationFailedEvent(tx *gorm.DB, payload proto.Message) error {
	payloadCast, ok := payload.(*events.OperationFailed)
	if !ok {
		return xerrors.Errorf("unexpected notification payload")
	}
	b.broadCastOperationStatus(tx, payloadCast.OperationId)
	return nil
}
