package tgbot

import (
	"context"
	"fmt"
	"sync"
	"time"

	tb "gopkg.in/tucnak/telebot.v2"
	"gorm.io/gorm"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
)

func (b *TGBot) checkAdmin(tx *gorm.DB, message *tb.Message) error {
	var numAdmins int64
	if err := tx.Model(model.TelegramAdmin{}).
		Where(model.TelegramAdmin{
			Username: message.Sender.Username,
		}).Count(&numAdmins).Error; err != nil {
		wrapped := xerrors.Errorf("unable to check for admin rights: %w", err)
		b.logger.Error(wrapped.Error(), log.Error(wrapped))
		return wrapTgError("Нет доступа", wrapped)
	}
	if numAdmins == 0 {
		return newTgError("Нет доступа")
	}
	return nil
}

func (b *TGBot) replyOnError(message *tb.Message, err error) {
	var tgError TGError
	var errorText string
	if xerrors.As(err, &tgError) {
		errorText = tgError.Message
	} else {
		errorText = "Неизвестная ошибка"
	}
	if _, sendError := b.tg.Send(message.Chat, errorText); sendError != nil {
		b.logger.Error("Unable to send error-reply to tg", log.Error(sendError))
	}
}

func (b *TGBot) replyWithText(message *tb.Message, text string, options ...interface{}) {
	if _, sendError := b.tg.Send(message.Chat, text, options...); sendError != nil {
		b.logger.Error("Unable to send text to tg", log.Error(sendError))
	}
}

func (b *TGBot) replyWithHTML(message *tb.Message, text string) {
	if _, sendError := b.tg.Send(message.Chat, text, &tb.SendOptions{ParseMode: tb.ModeHTML, DisableWebPagePreview: true}); sendError != nil {
		b.logger.Error("Unable to send html to tg", log.Error(sendError))
	}
}

func (b *TGBot) broadcastText(chatIDs []int64, text string, options ...interface{}) {
	for _, id := range chatIDs {
		if _, sendError := b.tg.Send(&tb.Chat{ID: id}, text, options...); sendError != nil {
			b.logger.Error("Unable to send text to tg", log.Error(sendError))
		}
	}
}

func (b *TGBot) broadcastHTML(chatIDs []int64, text string) {
	for _, id := range chatIDs {
		if _, sendError := b.tg.Send(&tb.Chat{ID: id}, text, &tb.SendOptions{ParseMode: tb.ModeHTML, DisableWebPagePreview: true}); sendError != nil {
			b.logger.Error("Unable to send text to tg", log.Error(sendError))
		}
	}
}

func (b *TGBot) broadCastRoomStayStatus(tx *gorm.DB, roomStayID uint64, action string) {
	rs, err := getRoomStayByID(tx, roomStayID)
	if err != nil {
		b.logger.Error("Unable to notify on roomstay event",
			log.UInt64("RoomStayID", roomStayID),
			log.Error(err))
		return
	}
	subs, err := getHotelSubscriptions(tx, rs.Room.Hotel, true)
	if err != nil {
		b.logger.Error("Unable to notify on roomstay event",
			log.UInt64("RoomStayID", roomStayID),
			log.Error(err))
		return
	}
	chatIDs := make([]int64, len(subs))
	for i, s := range subs {
		chatIDs[i] = s.ChatID
	}
	message := fmt.Sprintf("%s: номер %s %s", rs.Room.Hotel.Name, rs.Room.Name, action)
	b.broadcastText(chatIDs, message)
}

func (b *TGBot) broadCastRoomChange(tx *gorm.DB, oldRoomID uint64, newRoomID uint64) {
	oldRoom, err := getRoomByID(tx, oldRoomID)
	if err != nil {
		b.logger.Error("Unable to notify on room change event",
			log.UInt64("OldRoomID", oldRoomID),
			log.UInt64("NewRoomID", newRoomID),
			log.Error(err))
		return
	}
	newRoom, err := getRoomByID(tx, newRoomID)
	if err != nil {
		b.logger.Error("Unable to notify on room change event",
			log.UInt64("OldRoomID", oldRoomID),
			log.UInt64("NewRoomID", newRoomID),
			log.Error(err))
		return
	}
	subs, err := getHotelSubscriptions(tx, oldRoom.Hotel, true)
	if err != nil {
		b.logger.Error("Unable to notify on room change event",
			log.UInt64("OldRoomID", oldRoomID),
			log.UInt64("NewRoomID", newRoomID),
			log.Error(err))
		return
	}
	chatIDs := make([]int64, len(subs))
	for i, s := range subs {
		chatIDs[i] = s.ChatID
	}
	message := fmt.Sprintf("%s: гости переселены из номера %s в номер %s", oldRoom.Hotel.Name, oldRoom.Name, newRoom.Name)
	b.broadcastText(chatIDs, message)
}

func (b *TGBot) broadCastOperationStatus(tx *gorm.DB, operationID uint64) {
	o, err := getOperationByID(tx, operationID)
	if err != nil {
		b.logger.Error("Unable to notify on operation event",
			log.UInt64("OperationID", operationID),
			log.Error(err))
		return
	}
	subs, err := getHotelSubscriptions(tx, o.RoomStay.Room.Hotel, false)
	if err != nil {
		b.logger.Error("Unable to notify on operation event",
			log.UInt64("OperationID", operationID),
			log.Error(err))
		return
	}
	chatIDs := make([]int64, len(subs))
	for i, s := range subs {
		chatIDs[i] = s.ChatID
	}
	var opType string
	var opStatus string
	switch o.Type {
	case model.OperationTypeActivate:
		opType = "активации"
	case model.OperationTypeReset:
		opType = "сброса"
	default:
		b.logger.Warn("Unexpected operation type when notifying",
			log.UInt64("OperationID", operationID),
			log.String("Type", string(o.Type)))
		return
	}
	switch o.State {
	case model.OperationStateFailed:
		opStatus = "завершилась ошибкой"
	case model.OperationStateCancelled:
		opStatus = "отменена"
	default:
		b.logger.Warn("Unexpected operation state when notifying",
			log.UInt64("OperationID", operationID),
			log.String("State", string(o.State)))
		return
	}
	message := fmt.Sprintf("%s: номер %s: операция %s %s", o.RoomStay.Room.Hotel.Name, o.RoomStay.Room.Name, opType, opStatus)
	b.broadcastText(chatIDs, message)
}

func (b *TGBot) getReportForHotel(hotel *model.Hotel, force bool) (*HotelReport, error) {
	b.syncRoot.Lock()
	defer b.syncRoot.Unlock()
	report, exists := b.reports[hotel.ID]
	if !exists || force || report.generatedAt.Before(time.Now().Add(-1*b.cfg.ReportValidityDuration)) {
		db, err := b.pg.GetPrimary()
		if err != nil {
			return nil, xerrors.Errorf("unable to get db to generate report: %w", err)
		}
		report, err = buildRoomsReport(db, hotel, b.a4b)
		if err != nil {
			return nil, xerrors.Errorf("unable to generate report: %w", err)
		}
		b.reports[hotel.ID] = report
	}
	return report, nil
}

func loadA4BDataInChannel(ctx context.Context, a4b A4B, ch <-chan *RoomReportData) {
	for {
		select {
		case d := <-ch:
			if d == nil {
				return
			}
			if d.a4bID == "" {
				continue
			}
			d.a4bInfo, d.error = a4b.GetRoom(ctx, d.a4bID)
		case <-ctx.Done():
			return
		}
	}
}

func loadA4BData(ctx context.Context, data []*RoomReportData, a4b A4B, concurrency int) {
	wg := sync.WaitGroup{}
	wg.Add(concurrency)
	ch := make(chan *RoomReportData)
	for i := 0; i < concurrency; i++ {
		go func() {
			defer wg.Done()
			loadA4BDataInChannel(ctx, a4b, ch)
		}()
	}
	var needBreak bool
	for _, r := range data {
		select {
		case ch <- r:
			continue
		case <-ctx.Done():
			needBreak = true
		}
		if needBreak {
			break
		}
	}
	close(ch)
	wg.Wait()
}
