package tgbot

import (
	"database/sql"
	"fmt"
	"sort"
	"strconv"
	"strings"
	"time"
	"unicode/utf8"

	"gorm.io/gorm"

	"a.yandex-team.ru/library/go/core/xerrors"
	"a.yandex-team.ru/travel/budapest/metapms/internal/alice4business"
	"a.yandex-team.ru/travel/budapest/metapms/internal/model"
)

type RoomReportState int

const (
	RoomReportStateCheckOutToday RoomReportState = iota
	RoomReportStateCheckInToday
	RoomReportStateStayInProgress
	RoomReportStateEmpty
	RoomReportStateError
	RoomReportStateJustCheckedOut
)

var stateOrder = []RoomReportState{
	RoomReportStateJustCheckedOut,
	RoomReportStateCheckOutToday,
	RoomReportStateCheckInToday,
	RoomReportStateStayInProgress,
	RoomReportStateEmpty,
}

var pmsStateToPluralTitleMap = map[RoomReportState]string{
	RoomReportStateCheckOutToday:  "🛫 Выезд сегодня",
	RoomReportStateCheckInToday:   "🛬 Заезд сегодня",
	RoomReportStateStayInProgress: "🛌 Проживание",
	RoomReportStateEmpty:          "🛏 Не заселены",
	RoomReportStateJustCheckedOut: fmt.Sprintf("🧹 Недавно выселены (< %d минут)", int64(justCheckedOutInterval.Minutes())),
	RoomReportStateError:          "❌ Ошибка проверки",
}

var pmsStateToSingularTitleMap = map[RoomReportState]string{
	RoomReportStateCheckOutToday:  "🛫 Выезд сегодня",
	RoomReportStateCheckInToday:   "🛬 Заезд сегодня",
	RoomReportStateStayInProgress: "🛌 Проживание",
	RoomReportStateEmpty:          "🛏 Не заселен",
	RoomReportStateJustCheckedOut: "🧹 Недавно выселен",
	RoomReportStateError:          "❌ Ошибка проверки",
}

var roomStatusToDescriptionMap = map[alice4business.RoomStatus]string{
	alice4business.RoomActive:      "✅ Активирована",
	alice4business.RoomInactive:    "☑️ Сброшена",
	alice4business.RoomMixedStatus: "⚠️ По разному",
}

var deviceStatusToDescriptionMap = map[alice4business.DeviceStatus]string{
	alice4business.DeviceActive:   "✅ Активировано",
	alice4business.DeviceInactive: "☑️ Сброшено",
}

const resetInProgress = "⏳ Сбрасывается"
const resetFailed = "❌ Ошибка сброса"

var promoStatusMap = map[alice4business.RoomPromoStatus]string{
	alice4business.RoomPromoAvailable:    "доступен",
	alice4business.RoomPromoApplied:      "применен",
	alice4business.RoomPromoNotAvailable: "не доступен",
}

var opTypeMap = map[model.OperationType]string{
	model.OperationTypeReset:    "сброс",
	model.OperationTypeActivate: "активация",
}

const timeLayout = "15:04"
const dateLayout = "02.01"
const dateTimeLayout = "02.01 в 15:04"
const dateTimeLayoutSimple = "02.01 15:04"
const dateTimeSeconds = "02.01 15:04:05"
const justCheckedOutInterval = time.Hour * 2
const longRunningOperationTimeout = time.Hour * 2

const travellinePrefix = "https://www.travelline.ru/secure/webpms/extranet/#/WebPmsBookingcardV2/%s"

type RoomReportData struct {
	roomID           uint
	name             string
	hotelName        string
	hotelLocation    *time.Location
	pastStay         *model.RoomStay
	currentStay      *model.RoomStay
	nextStay         *model.RoomStay
	a4bID            string
	a4bInfo          *alice4business.RoomInfo
	runningOperation *model.Operation
	error            error
}

type HotelReport struct {
	roomReportData []*RoomReportData
	hotel          *model.Hotel
	generatedAt    time.Time
}

func (rrd *RoomReportData) GetState() RoomReportState {
	now := time.Now()
	tomorrow := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, rrd.hotelLocation).AddDate(0, 0, 1)
	if rrd.error != nil {
		return RoomReportStateError
	}
	if rrd.pastStay != nil && rrd.pastStay.ActualCheckOutDateTime.Valid && rrd.currentStay == nil {
		if rrd.pastStay.ActualCheckOutDateTime.Time.After(now.Add(-1 * justCheckedOutInterval)) {
			return RoomReportStateJustCheckedOut
		}
	}
	if rrd.currentStay != nil {
		if rrd.currentStay.CheckOutDateTime.Valid && rrd.currentStay.CheckOutDateTime.Time.Before(tomorrow) {
			return RoomReportStateCheckOutToday
		} else {
			return RoomReportStateStayInProgress
		}
	} else {
		if rrd.nextStay != nil && rrd.nextStay.CheckInDateTime.Valid && rrd.nextStay.CheckInDateTime.Time.Before(tomorrow) {
			return RoomReportStateCheckInToday
		} else {
			return RoomReportStateEmpty
		}
	}
}

func getRoomStatusString(status alice4business.RoomStatus, inProgress bool) string {
	switch status {
	case alice4business.RoomReset:
		if inProgress {
			return resetInProgress
		} else {
			return resetFailed
		}
	default:
		return roomStatusToDescriptionMap[status]
	}
}

func getDeviceStatusString(status alice4business.DeviceStatus, inProgress bool) string {
	switch status {
	case alice4business.DeviceReset:
		if inProgress {
			return resetInProgress
		} else {
			return resetFailed
		}
	default:
		return deviceStatusToDescriptionMap[status]
	}
}

func (rrd *RoomReportData) getAliceString() string {
	var targetStatus alice4business.RoomStatus
	switch rrd.GetState() {
	case RoomReportStateCheckInToday, RoomReportStateEmpty:
		targetStatus = alice4business.RoomInactive
	case RoomReportStateCheckOutToday, RoomReportStateStayInProgress:
		targetStatus = alice4business.RoomActive
	case RoomReportStateJustCheckedOut:
		if rrd.currentStay != nil {
			targetStatus = alice4business.RoomActive
		} else {
			targetStatus = alice4business.RoomInactive
		}
	}

	if rrd.a4bInfo == nil {
		return "❓"
	}
	var devicesInProgress bool
	var someDevicesActive bool
	for _, d := range rrd.a4bInfo.Devices {
		if d.InProgress {
			devicesInProgress = true
		}
		if d.Status == alice4business.DeviceActive {
			someDevicesActive = true
		}
	}

	if rrd.a4bInfo.InProgress || devicesInProgress {
		if targetStatus == alice4business.RoomActive && rrd.a4bInfo.Status == alice4business.RoomMixedStatus && someDevicesActive {
			return "⚠️"
		} else {
			return "⏳"
		}
	}
	if rrd.a4bInfo.Status == alice4business.RoomMixedStatus {
		return "⚠️"
	}
	if rrd.runningOperation != nil {
		return "⏳"
	}
	if rrd.a4bInfo.Status == targetStatus {
		if targetStatus == alice4business.RoomActive {
			var checkLogin string
			for _, d := range rrd.a4bInfo.Devices {
				if d.HasCustomUser {
					checkLogin = "😱"
				}
			}
			return "✅" + checkLogin
		} else {
			return "☑️"
		}
	} else {
		return "❌"
	}
}

func (rrd *RoomReportData) getJustCheckedOutStatus(html bool) string {
	var parts []string

	checkedOutTime := rrd.pastStay.ActualCheckOutDateTime.Time.Format(timeLayout)
	checkedOutString := fmt.Sprintf("выехал в %s", checkedOutTime)
	if html {
		url := fmt.Sprintf(travellinePrefix, rrd.pastStay.Booking.TravellineNumber)
		checkedOutString = fmt.Sprintf("<a href='%s'>%s</a>", url, checkedOutString)
	}
	parts = append(parts, checkedOutString)
	if rrd.currentStay != nil && rrd.currentStay.ActualCheckInDateTime.Valid {
		checkedInTime := rrd.currentStay.ActualCheckInDateTime.Time.Format(timeLayout)
		checkedInString := fmt.Sprintf("заехал в %s", checkedInTime)
		if html {
			url := fmt.Sprintf(travellinePrefix, rrd.currentStay.Booking.TravellineNumber)
			checkedInString = fmt.Sprintf("<a href='%s'>%s</a>", url, checkedInString)
		}
		parts = append(parts, checkedInString)
	} else if rrd.nextStay != nil && rrd.nextStay.CheckInDateTime.Valid {
		var layout string
		now := time.Now()
		tomorrow := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, rrd.hotelLocation).AddDate(0, 0, 1)
		if rrd.nextStay.CheckInDateTime.Time.Before(tomorrow) {
			layout = timeLayout
		} else {
			layout = dateTimeLayout
		}
		nextCheckInTime := rrd.nextStay.CheckInDateTime.Time.Format(layout)
		nextCheckInString := fmt.Sprintf("следующий %s", nextCheckInTime)
		if html {
			url := fmt.Sprintf(travellinePrefix, rrd.nextStay.Booking.TravellineNumber)
			nextCheckInString = fmt.Sprintf("<a href='%s'>%s</a>", url, nextCheckInString)
		}
		parts = append(parts, nextCheckInString)
	}
	aliceString := fmt.Sprintf("устройства %s", rrd.getAliceString())
	parts = append(parts, aliceString)
	return fmt.Sprintf("%s: %s", rrd.name, strings.Join(parts, ", "))
}

func (rrd *RoomReportData) getCheckoutTodayStatus(html bool) string {
	var parts []string

	checkoutTime := rrd.currentStay.CheckInDateTime.Time.Format(timeLayout)
	checkoutString := fmt.Sprintf("выезд в %s", checkoutTime)
	if html {
		url := fmt.Sprintf(travellinePrefix, rrd.currentStay.Booking.TravellineNumber)
		checkoutString = fmt.Sprintf("<a href='%s'>%s</a>", url, checkoutString)
	}

	parts = append(parts, checkoutString)

	if rrd.nextStay != nil && rrd.nextStay.CheckInDateTime.Valid {
		var nextCheckinTime string
		now := time.Now()
		tomorrow := time.Date(now.Year(), now.Month(), now.Day(), 0, 0, 0, 0, rrd.hotelLocation).AddDate(0, 0, 1)
		var today bool
		if rrd.nextStay.CheckInDateTime.Time.Before(tomorrow) {
			today = true
			nextCheckinTime = fmt.Sprintf("в %s", rrd.nextStay.CheckInDateTime.Time.Format(timeLayout))
		} else {
			nextCheckinTime = rrd.nextStay.CheckInDateTime.Time.Format(dateLayout)
		}
		nextCheckinString := fmt.Sprintf("следующий заезд %s", nextCheckinTime)
		if html {
			if rrd.nextStay.Booking != nil {
				url := fmt.Sprintf(travellinePrefix, rrd.nextStay.Booking.TravellineNumber)
				nextCheckinString = fmt.Sprintf("<a href='%s'>%s</a>", url, nextCheckinString)
			}
			if today {
				nextCheckinString = fmt.Sprintf("<b>%s</b>", nextCheckinString)
			}
		}
		parts = append(parts, nextCheckinString)
	}
	aliceString := fmt.Sprintf("устройства %s", rrd.getAliceString())

	parts = append(parts, aliceString)

	return fmt.Sprintf("%s: %s", rrd.name, strings.Join(parts, ", "))
}

func (rrd *RoomReportData) getCheckinTodayStatus(html bool) string {
	nextCheckinString := fmt.Sprintf("заезд в %s", rrd.nextStay.CheckInDateTime.Time.Format(timeLayout))
	if html {
		url := fmt.Sprintf(travellinePrefix, rrd.nextStay.Booking.TravellineNumber)
		nextCheckinString = fmt.Sprintf("<a href='%s'>%s</a>", url, nextCheckinString)
	}
	aliceString := fmt.Sprintf("устройства %s", rrd.getAliceString())

	return fmt.Sprintf("%s: %s, %s", rrd.name, nextCheckinString, aliceString)
}

func (rrd *RoomReportData) getStayInProgressStatus(html bool) string {
	checkoutString := fmt.Sprintf("выезд %s", rrd.currentStay.CheckOutDateTime.Time.Format(dateLayout))
	if html {
		url := fmt.Sprintf(travellinePrefix, rrd.currentStay.Booking.TravellineNumber)
		checkoutString = fmt.Sprintf("<a href='%s'>%s</a>", url, checkoutString)
	}
	aliceString := fmt.Sprintf("устройства %s", rrd.getAliceString())
	return fmt.Sprintf("%s: %s, %s", rrd.name, checkoutString, aliceString)
}

func (rrd *RoomReportData) getEmptyStatus(html bool) string {
	var nextCheckinString string
	if rrd.nextStay != nil {
		nextCheckinString = fmt.Sprintf("заезд %s", rrd.nextStay.CheckInDateTime.Time.Format(dateTimeLayout))
		if html {
			if rrd.nextStay.Booking != nil {
				url := fmt.Sprintf(travellinePrefix, rrd.nextStay.Booking.TravellineNumber)
				nextCheckinString = fmt.Sprintf("<a href='%s'>%s</a>", url, nextCheckinString)
			}
		}
		nextCheckinString += ", "
	}
	aliceString := fmt.Sprintf("устройства %s", rrd.getAliceString())

	return fmt.Sprintf("%s: %s%s", rrd.name, nextCheckinString, aliceString)
}

func (rrd *RoomReportData) isLongRunning() bool {
	return rrd.runningOperation != nil && rrd.runningOperation.CreatedAt.Before(time.Now().Add(-1*longRunningOperationTimeout))
}

func (rrd *RoomReportData) getNameForRoomList() string {
	r, _ := utf8.DecodeRuneInString(pmsStateToSingularTitleMap[rrd.GetState()])
	alice := rrd.getAliceString()
	if alice == "✅" || alice == "☑️" {
		alice = ""
	}
	return fmt.Sprintf("%s %s %s", rrd.name, string(r), alice)
}

func (rrd *RoomReportData) getDetailsDescription(html bool) string {
	var parts []string
	title := fmt.Sprintf("Номер #%s (отель %s) на %s", rrd.name, rrd.hotelName, time.Now().Format(dateTimeSeconds))
	if html {
		title = fmt.Sprintf("<b><u>%s</u></b>", title)
	}
	statePrefix := "Состояние"
	if html {
		statePrefix = fmt.Sprintf("<b>%s</b>", statePrefix)
	}
	state := fmt.Sprintf("%s: %s", statePrefix, pmsStateToSingularTitleMap[rrd.GetState()])
	parts = append(parts, title, "", state, "")
	if rrd.pastStay != nil && rrd.pastStay.ActualCheckInDateTime.Valid && rrd.pastStay.ActualCheckOutDateTime.Valid {
		stayPrefix := "Прошлый гость"
		if html {
			stayPrefix = fmt.Sprintf("<b>%s</b>", stayPrefix)
		}
		stay := fmt.Sprintf("%s: заехал %s, выехал %s",
			stayPrefix,
			rrd.pastStay.ActualCheckInDateTime.Time.Format(dateTimeLayout),
			rrd.pastStay.ActualCheckOutDateTime.Time.Format(dateTimeLayout),
		)
		parts = append(parts, stay)
	}
	if rrd.currentStay != nil && rrd.currentStay.ActualCheckInDateTime.Valid && rrd.currentStay.CheckOutDateTime.Valid {
		stayPrefix := "Текущий гость"
		if html {
			stayPrefix = fmt.Sprintf("<b>%s</b>", stayPrefix)
		}
		stay := fmt.Sprintf("%s: заехал %s, выедет %s",
			stayPrefix,
			rrd.currentStay.ActualCheckInDateTime.Time.Format(dateTimeLayout),
			rrd.currentStay.CheckOutDateTime.Time.Format(dateTimeLayout),
		)
		parts = append(parts, stay)
	}
	if rrd.nextStay != nil && rrd.nextStay.CheckInDateTime.Valid && rrd.nextStay.CheckOutDateTime.Valid {
		stayPrefix := "Следующий гость"
		if html {
			stayPrefix = fmt.Sprintf("<b>%s</b>", stayPrefix)
		}
		stay := fmt.Sprintf("%s: должен заехать %s, выехать %s",
			stayPrefix,
			rrd.nextStay.CheckInDateTime.Time.Format(dateTimeLayout),
			rrd.nextStay.CheckOutDateTime.Time.Format(dateTimeLayout),
		)
		parts = append(parts, stay)
	}
	if rrd.a4bInfo != nil {
		devicesTitle := "Устройства:"
		if html {
			devicesTitle = fmt.Sprintf("<b>%s</b>", devicesTitle)
		}
		parts = append(parts, "", devicesTitle)
		roomState := fmt.Sprintf("Комната: %s, плюс %s",
			getRoomStatusString(rrd.a4bInfo.Status, rrd.a4bInfo.InProgress),
			promoStatusMap[rrd.a4bInfo.PromoStatus])
		parts = append(parts, roomState)
		for _, d := range rrd.a4bInfo.Devices {
			var hasPlus string
			if d.HasPlus {
				hasPlus = "плюс есть"
			} else {
				hasPlus = "плюса нет"
			}
			var network string
			if d.Online {
				network = "📶"
			} else {
				network = "🚫"
			}
			dState := fmt.Sprintf("%s %s: %s, %s",
				network, d.Note, getDeviceStatusString(d.Status, d.InProgress), hasPlus)
			if d.HasCustomUser && d.Status == alice4business.DeviceActive {
				extra := "пользовательский логин! 😱"
				if html {
					extra = fmt.Sprintf("<b>%s</b>", extra)
				}
				dState += ", " + extra
			}
			parts = append(parts, dState)
		}
	}
	if rrd.runningOperation != nil {
		opTitle := "Теущая операция: "
		if html {
			opTitle = fmt.Sprintf("<b>%s</b>", opTitle)
		}
		opTitle += opTypeMap[rrd.runningOperation.Type]
		opRunningSince := fmt.Sprintf("Запущена с %s", rrd.runningOperation.CreatedAt.Format(dateTimeLayoutSimple))
		parts = append(parts, "", opTitle, opRunningSince)
	}

	return strings.Join(parts, "\r\n")
}

func buildRoomsReport(tx *gorm.DB, hotel *model.Hotel, a4b A4B) (*HotelReport, error) {
	var rooms []model.Room
	loc, err := hotel.GetLocation()
	if err != nil {
		return nil, xerrors.Errorf("unable to get hotel location: %w", err)
	}
	if err := tx.Model(model.Room{}).
		Where(model.Room{HotelID: hotel.ID, ReportsEnabled: sql.NullBool{
			Bool:  true,
			Valid: true,
		}}).
		Joins("RoomBinding").
		Find(&rooms).Error; err != nil {
		return nil, xerrors.Errorf("unable to fetch rooms for report: %w", err)
	}
	past, err := getPastStays(tx, hotel)
	if err != nil {
		return nil, err
	}
	active, err := getActiveStays(tx, hotel)
	if err != nil {
		return nil, err
	}
	upcoming, err := getUpcomingStays(tx, hotel)
	if err != nil {
		return nil, err
	}
	runningOperations, err := getRunningOperations(tx, hotel)
	if err != nil {
		return nil, err
	}

	allRooms := make([]*RoomReportData, len(rooms))
	for i, room := range rooms {
		rrd := &RoomReportData{
			roomID:        room.ID,
			hotelName:     hotel.Name,
			name:          room.Name,
			hotelLocation: loc,
		}
		if room.RoomBinding != nil {
			rrd.a4bID = room.RoomBinding.A4BRoomID
		}
		if stay, exists := past[room.ID]; exists {
			rrd.pastStay = stay
		}
		if stay, exists := active[room.ID]; exists {
			rrd.currentStay = stay
		}
		if stay, exists := upcoming[room.ID]; exists {
			rrd.nextStay = stay
		}
		if operation, exist := runningOperations[room.ID]; exist {
			rrd.runningOperation = operation
		}
		allRooms[i] = rrd
	}
	loadA4BData(tx.Statement.Context, allRooms, a4b, 30)
	sort.Slice(allRooms, func(i, j int) bool {
		left := allRooms[i]
		right := allRooms[j]
		leftNum, convErr1 := strconv.Atoi(left.name)
		rightNum, convErr2 := strconv.Atoi(right.name)
		if convErr1 == nil && convErr2 == nil {
			return leftNum < rightNum
		} else {
			return left.name < right.name
		}
	})

	result := HotelReport{hotel: hotel, roomReportData: allRooms, generatedAt: time.Now()}
	return &result, nil
}

func (hr *HotelReport) getRoomReport(roomID uint) *RoomReportData {
	for _, d := range hr.roomReportData {
		if d.roomID == roomID {
			return d
		}
	}
	return nil
}

func (hr *HotelReport) printRoomsReport(html bool) string {
	items := make(map[RoomReportState][]*RoomReportData, len(hr.roomReportData))
	for _, r := range hr.roomReportData {
		state := r.GetState()
		sl := items[state]
		sl = append(sl, r)
		items[state] = sl
	}
	var results []string
	header := fmt.Sprintf("Отель «%s», состояние номеров на %s:", hr.hotel.Name,
		hr.hotel.Sync.SyncedTill.Format(dateTimeSeconds))
	if html {
		header = fmt.Sprintf("<b><u>%s</u></b>", header)
	}
	results = append(results, header, "")
	for _, state := range stateOrder {
		if list, exist := items[state]; exist && len(list) > 0 {
			blockTitle := fmt.Sprintf("%s:", pmsStateToPluralTitleMap[state])
			if html {
				blockTitle = fmt.Sprintf("<b>%s</b>", strings.Replace(blockTitle, "<", "&lt;", -1))
			}
			results = append(results, blockTitle)
			for _, rrd := range list {
				var descr string
				switch state {
				case RoomReportStateCheckOutToday:
					descr = rrd.getCheckoutTodayStatus(html)
				case RoomReportStateCheckInToday:
					descr = rrd.getCheckinTodayStatus(html)
				case RoomReportStateStayInProgress:
					descr = rrd.getStayInProgressStatus(html)
				case RoomReportStateEmpty:
					descr = rrd.getEmptyStatus(html)
				case RoomReportStateJustCheckedOut:
					descr = rrd.getJustCheckedOutStatus(html)
				}
				results = append(results, descr)
			}
			results = append(results, "")
		}
	}

	var longRunningNames []string
	for _, g := range items {
		for _, gg := range g {
			if gg.isLongRunning() {
				longRunningNames = append(longRunningNames, gg.name)
			}
		}
	}
	if len(longRunningNames) > 0 {
		title := fmt.Sprintf("⏳ Долго сбрасываются (>%d минут):", int64(longRunningOperationTimeout.Minutes()))
		if html {
			title = fmt.Sprintf("<b>%s</b>", title)
		}
		results = append(results, title)
		sort.Slice(longRunningNames, func(i, j int) bool {
			left := longRunningNames[i]
			right := longRunningNames[j]
			leftNum, convErr1 := strconv.Atoi(left)
			rightNum, convErr2 := strconv.Atoi(right)
			if convErr1 == nil && convErr2 == nil {
				return leftNum < rightNum
			} else {
				return left < right
			}
		})
		results = append(results, strings.Join(longRunningNames, ", "))
		results = append(results, "")
	}

	if erred, exists := items[RoomReportStateError]; exists && len(erred) > 0 {
		var erredRooms []string
		for _, e := range erred {
			erredRooms = append(erredRooms, e.name)
		}
		results = append(results, fmt.Sprintf("❌ Произошли ошибки при проверке номеров %s", strings.Join(erredRooms, ", ")))
	}
	return strings.Join(results, "\r\n")
}
